package repositories

import (
	"context"
	"sync/atomic"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type RouteInfosRepository struct {
	routeInfoByRoute atomic.Value
	dataGetter       func() ([]models.RouteInfo, error)
	logger           log.Logger
}

func NewRouteInfosRepository(logger log.Logger, dataGetter func() ([]models.RouteInfo, error)) *RouteInfosRepository {
	return &RouteInfosRepository{dataGetter: dataGetter, logger: logger}
}

func (r *RouteInfosRepository) Update() error {
	r.logger.Info("precaching route_infos")
	data, err := r.dataGetter()
	if err != nil {
		return err
	}
	var routeInfoByRoute = make(map[models.Route]models.RouteInfo)
	for _, routeInfo := range data {
		key := models.Route{FromID: routeInfo.FromID, ToID: routeInfo.ToID}
		routeInfoByRoute[key] = routeInfo
	}
	r.routeInfoByRoute.Store(routeInfoByRoute)
	r.logger.Info("route_infos precached")
	return nil
}

func (r *RouteInfosRepository) Get(ctx context.Context, fromID, toID uint32) (
	*models.RouteInfo,
	error,
) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.storage.RouteInfosRepository:Get")
	defer span.Finish()

	d := models.Route{FromID: fromID, ToID: toID}
	if value, found := r.routeInfoByRoute.Load().(map[models.Route]models.RouteInfo)[d]; found {
		return &value, nil
	}
	return nil, models.ErrorNotFound
}
