package repositories

import (
	"context"
	"sync/atomic"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type RoutesRepository struct {
	routesSet  atomic.Value
	routes     atomic.Value
	dataGetter func() ([]models.Route, error)
	logger     log.Logger
}

func NewRoutesRepository(logger log.Logger, dataGetter func() ([]models.Route, error)) *RoutesRepository {
	return &RoutesRepository{dataGetter: dataGetter, logger: logger}
}

func (r *RoutesRepository) Update() error {
	r.logger.Info("precaching routes")
	data, err := r.dataGetter()
	if err != nil {
		return err
	}
	var routesSet = make(map[models.Route]struct{})
	var routes = make([]models.Route, 0)
	for _, route := range data {
		routesSet[route] = struct{}{}
		routes = append(routes, route)
	}
	r.routesSet.Store(routesSet)
	r.routes.Store(routes)
	r.logger.Info("routes precached")
	return nil
}

func (r *RoutesRepository) Contains(ctx context.Context, route models.Route) bool {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.storage.RoutesRepository:Contains")
	defer span.Finish()

	_, found := r.routesSet.Load().(map[models.Route]struct{})[route]
	return found
}

func (r *RoutesRepository) GetAll() []models.Route {
	return r.routes.Load().([]models.Route)
}
