package repositories

import (
	"context"
	"sync/atomic"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/pkg/tables"
)

type SettlementPopularity struct {
	SettlementID uint32
	RedirNumber  uint32
}

type SettlementPopularityRepository struct {
	settlementPopularity atomic.Value
	dataGetter           func() ([]tables.DirectionPopularityEntry, error)
	logger               log.Logger
}

func NewSettlementPopularityRepository(
	logger log.Logger, dataGetter func() (
		[]tables.DirectionPopularityEntry,
		error,
	),
) *SettlementPopularityRepository {
	return &SettlementPopularityRepository{dataGetter: dataGetter, logger: logger}
}

func (r *SettlementPopularityRepository) Update() error {
	r.logger.Info("precaching settlement_popularity")
	data, err := r.dataGetter()
	if err != nil {
		return err
	}
	var popularityByID = make(map[uint32]uint32)
	for _, info := range data {
		popularityByID[info.SettlementToID] += info.RedirNumber
	}
	r.settlementPopularity.Store(popularityByID)
	r.logger.Info("settlement_popularity precached")
	return nil
}

func (r *SettlementPopularityRepository) Get(ctx context.Context, settlementID uint32) (uint32, bool) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.storage.SettlementPopularityRepository:Get")
	defer span.Finish()

	value, ok := r.settlementPopularity.Load().(map[uint32]uint32)[settlementID]
	return value, ok
}
