package repositories

import (
	"context"
	"sync/atomic"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type TopAirlinesRepository struct {
	topAirlinesByRoute atomic.Value
	dataGetter         func() ([]models.TopAirlines, error)
	logger             log.Logger
}

func NewTopAirlinesRepository(logger log.Logger, dataGetter func() ([]models.TopAirlines, error)) *TopAirlinesRepository {
	return &TopAirlinesRepository{dataGetter: dataGetter, logger: logger}
}

func (r *TopAirlinesRepository) Update() error {
	r.logger.Info("precaching top_airlines")
	data, err := r.dataGetter()
	if err != nil {
		return err
	}
	var topAirlinesByRoute = make(map[models.Route]models.TopAirlines)
	for _, topAirlines := range data {
		key := models.Route{FromID: topAirlines.FromID, ToID: topAirlines.ToID, NationalVersion: topAirlines.NationalVersion}
		topAirlinesByRoute[key] = topAirlines
	}
	r.topAirlinesByRoute.Store(topAirlinesByRoute)
	r.logger.Info("top_airlines precached")
	return nil
}

func (r *TopAirlinesRepository) Get(ctx context.Context, fromID, toID uint32, nationalVersion string) (
	[]uint32,
	error,
) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.storage.TopAirlinesRepository:Get")
	defer span.Finish()

	d := models.Route{FromID: fromID, ToID: toID, NationalVersion: nationalVersion}
	if value, found := r.topAirlinesByRoute.Load().(map[models.Route]models.TopAirlines)[d]; found {
		return value.TopAirlines, nil
	}
	return nil, models.ErrorNotFound
}
