package tables

import (
	"context"
	"fmt"
	"strconv"
	"strings"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/landingcityto"
)

type CityToNearestCitiesTable struct {
	sessionPool        *table.SessionPool
	transactionControl *table.TransactionControl
	tableName          string
	batchSize          uint64
}

func NewCityToNearestCitiesTable(sessionPool *table.SessionPool, cfg Config) *CityToNearestCitiesTable {
	return &CityToNearestCitiesTable{
		sessionPool: sessionPool,
		transactionControl: table.TxControl(
			table.BeginTx(table.WithStaleReadOnly()),
			table.CommitTx(),
		),
		tableName: cfg.CityToNearestCitiesTableName,
		batchSize: cfg.ReadTableBatchSize,
	}
}

func (t *CityToNearestCitiesTable) GetAll() (cityToNearestCities []landingcityto.CityToNearestCities, err error) {
	lastToID := uint32(0)
	lastNationalVersion := ""
	var page CityToNearestCitiesEntries
	for {
		page, err = t.getNextPage(
			t.batchSize,
			lastToID,
			lastNationalVersion,
		)
		if err != nil || len(page) == 0 {
			return
		}
		for _, row := range page {
			nearestCities := landingcityto.CityToNearestCities{
				ToID:            row.ToID,
				NationalVersion: row.NationalVersion,
				NearestCityIds:  getIds(row.NearestCityIds),
			}
			cityToNearestCities = append(cityToNearestCities, nearestCities)
		}
		lastValue := cityToNearestCities[len(cityToNearestCities)-1]
		lastToID = lastValue.ToID
		lastNationalVersion = lastValue.NationalVersion
	}
}

func (t *CityToNearestCitiesTable) getNextPage(
	limit uint64,
	lastToID uint32,
	lastNationalVersion string,
) (page CityToNearestCitiesEntries, err error) {
	var query = fmt.Sprintf(
		`
			DECLARE $limit AS Uint64;
			DECLARE $last_to_id AS Uint32;
			DECLARE $last_national_version AS Utf8;

			SELECT * FROM %[1]s
			WHERE
				(to_id, national_version) > ($last_to_id, $last_national_version)
			ORDER BY to_id, national_version
			LIMIT $limit;
		`, t.tableName,
	)

	var res *table.Result

	err = table.Retry(
		context.Background(), t.sessionPool,
		table.OperationFunc(
			func(ctx context.Context, s *table.Session) (err error) {
				preparedStatement, err := s.Prepare(ctx, query)
				if err != nil {
					return err
				}
				_, res, err = preparedStatement.Execute(
					ctx, t.transactionControl,
					table.NewQueryParameters(
						table.ValueParam("$limit", ydb.Uint64Value(limit)),
						table.ValueParam("$last_to_id", ydb.Uint32Value(lastToID)),
						table.ValueParam("$last_national_version", ydb.UTF8Value(lastNationalVersion)),
					),
				)
				return
			},
		),
	)

	if err != nil {
		return
	}
	if err = res.Err(); err != nil {
		return
	}

	if !res.NextSet() || !res.HasNextRow() {
		return
	}

	err = (&page).Scan(res)
	return
}

func getIds(comaSeparatedIds string) []uint32 {
	tmp := strings.Split(comaSeparatedIds, ",")
	ids := make([]uint32, 0, len(tmp))
	for _, id := range tmp {
		id64, err := strconv.ParseUint(id, 10, 32)
		if err == nil {
			ids = append(ids, uint32(id64))
		}
	}
	return ids
}
