package tables

import (
	"context"
	"fmt"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/landingcityto"
)

type CityToRouteCrosslinksTable struct {
	sessionPool        *table.SessionPool
	transactionControl *table.TransactionControl
	tableName          string
	batchSize          int
}

func NewCityToRouteCrosslinksTable(sessionPool *table.SessionPool, cfg Config) *CityToRouteCrosslinksTable {
	return &CityToRouteCrosslinksTable{
		sessionPool: sessionPool,
		transactionControl: table.TxControl(
			table.BeginTx(table.WithStaleReadOnly()),
			table.CommitTx(),
		),
		tableName: cfg.CityToRouteCrosslinksTableName,
		batchSize: 1000,
	}
}

func (t *CityToRouteCrosslinksTable) GetAll() (cityToRouteCrosslinks []landingcityto.CityToRouteCrosslink, err error) {
	lastToID := uint32(0)
	lastNationalVersion := ""
	lastPosition := uint8(0)
	var page CityToRouteCrosslinkEntries
	for {
		page, err = t.getNextPage(
			t.batchSize,
			lastToID,
			lastPosition,
			lastNationalVersion,
		)
		if err != nil || len(page) == 0 {
			return
		}
		for _, row := range page {
			var price *uint32
			if p, ok := row.Price.Get(); ok {
				price = ptr.Uint32(p)
			}
			var currency *string
			if c, ok := row.Currency.Get(); ok {
				currency = ptr.String(c)
			}
			var date *string
			if d, ok := row.Date.Get(); ok {
				date = ptr.String(d)
			}
			crosslink := landingcityto.CityToRouteCrosslink{
				ToID:            row.ToID,
				NationalVersion: row.NationalVersion,
				CrosslinkFromID: row.CrosslinkFromID,
				CrosslinkToID:   row.CrosslinkToID,
				Price:           price,
				Currency:        currency,
				Date:            date,
			}
			lastPosition = row.Position
			cityToRouteCrosslinks = append(cityToRouteCrosslinks, crosslink)
		}
		lastValue := cityToRouteCrosslinks[len(cityToRouteCrosslinks)-1]
		lastToID = lastValue.ToID
		lastNationalVersion = lastValue.NationalVersion
	}
}

func (t *CityToRouteCrosslinksTable) getNextPage(
	limit int,
	lastToID uint32,
	lastPosition uint8,
	lastNationalVersion string,
) (page CityToRouteCrosslinkEntries, err error) {
	var query = fmt.Sprintf(
		`
			DECLARE $limit AS Uint64;
			DECLARE $last_to_id AS Uint32;
			DECLARE $last_national_version AS Utf8;
			DECLARE $last_position AS Uint8;

			SELECT * FROM %[1]s
			WHERE
				(to_id, national_version, position) > ($last_to_id, $last_national_version, $last_position)
			ORDER BY to_id, national_version, position
			LIMIT $limit;
		`, t.tableName,
	)

	var res *table.Result

	err = table.Retry(
		context.Background(), t.sessionPool,
		table.OperationFunc(
			func(ctx context.Context, s *table.Session) (err error) {
				preparedStatement, err := s.Prepare(ctx, query)
				if err != nil {
					return err
				}
				_, res, err = preparedStatement.Execute(
					ctx, t.transactionControl,
					table.NewQueryParameters(
						table.ValueParam("$limit", ydb.Uint64Value(uint64(limit))),
						table.ValueParam("$last_to_id", ydb.Uint32Value(lastToID)),
						table.ValueParam("$last_national_version", ydb.UTF8Value(lastNationalVersion)),
						table.ValueParam("$last_position", ydb.Uint8Value(lastPosition)),
					),
				)
				return
			},
		),
	)

	if err != nil {
		return
	}
	if err = res.Err(); err != nil {
		return
	}

	if !res.NextSet() || !res.HasNextRow() {
		return
	}

	err = (&page).Scan(res)
	return
}
