package tables

import (
	"context"
	"fmt"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
)

type DirectionPopularityTable struct {
	sessionPool        *table.SessionPool
	transactionControl *table.TransactionControl
	tableName          string
	batchSize          int
}

func NewDirectionPopularityTable(sessionPool *table.SessionPool, cfg Config) *DirectionPopularityTable {
	return &DirectionPopularityTable{
		sessionPool: sessionPool,
		tableName:   cfg.DirectionPopularityTableName,
		batchSize:   1000,
		transactionControl: table.TxControl(
			table.BeginTx(table.WithStaleReadOnly()),
			table.CommitTx(),
		),
	}
}

func (t *DirectionPopularityTable) GetAll() ([]DirectionPopularityEntry, error) {
	lastFromID := uint32(0)
	lastToID := uint32(0)
	var result []DirectionPopularityEntry
	for {
		page, err := t.getNextPage(t.batchSize, lastFromID, lastToID)
		if err != nil || len(page) == 0 {
			return result, err
		}
		for _, row := range page {
			result = append(result, row)
		}
		lastRes := result[len(result)-1]
		lastFromID = lastRes.SettlementFromID
		lastToID = lastRes.SettlementToID
	}
}

func (t *DirectionPopularityTable) getNextPage(limit int, lastFromID, lastToID uint32) (DirectionPopularityEntries,
	error) {
	var query = fmt.Sprintf(
		`
			DECLARE $limit AS Uint64;
			DECLARE $last_from_id AS Uint32;
			DECLARE $last_to_id AS Uint32;

			SELECT * FROM %[1]s
			WHERE (settlement_from_id, settlement_to_id) > ($last_from_id, $last_to_id)
			ORDER BY settlement_from_id, settlement_to_id
			LIMIT $limit;
		`, t.tableName,
	)

	var res *table.Result
	err := table.Retry(
		context.Background(), t.sessionPool,
		table.OperationFunc(
			func(ctx context.Context, s *table.Session) (err error) {
				preparedStatement, err := s.Prepare(ctx, query)
				if err != nil {
					return err
				}
				_, res, err = preparedStatement.Execute(
					ctx, t.transactionControl,
					table.NewQueryParameters(
						table.ValueParam("$limit", ydb.Uint64Value(uint64(limit))),
						table.ValueParam("$last_from_id", ydb.Uint32Value(lastFromID)),
						table.ValueParam("$last_to_id", ydb.Uint32Value(lastToID)),
					),
				)
				return
			},
		),
	)
	if err != nil {
		return nil, err
	}
	if err = res.Err(); err != nil {
		return nil, err
	}

	if !res.NextResultSet(context.Background()) || !res.HasNextRow() {
		return DirectionPopularityEntries{}, nil
	}

	var result = DirectionPopularityEntries{}
	err = (&result).Scan(res)
	return result, err
}
