package tables

import (
	"context"
	"fmt"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type MinPricesByAirlineTable struct {
	sessionPool        *table.SessionPool
	transactionControl *table.TransactionControl
	tableName          string
	batchSize          int
}

func NewMinPricesByAirlineTable(sessionPool *table.SessionPool, cfg Config) *MinPricesByAirlineTable {
	return &MinPricesByAirlineTable{
		sessionPool: sessionPool,
		transactionControl: table.TxControl(
			table.BeginTx(table.WithStaleReadOnly()),
			table.CommitTx(),
		),
		tableName: cfg.MinPricesByAirlineTableName,
		batchSize: 1000,
	}
}

func (t *MinPricesByAirlineTable) GetAll() (minPricesByAirloines []models.MinPriceByAirline, err error) {
	lastFromID := uint32(0)
	lastToID := uint32(0)
	lastAirlineID := uint32(0)
	lastNationalVersion := ""
	var page MinPriceByAirlineEntries
	for {
		page, err = t.getNextPage(t.batchSize, lastFromID, lastToID, lastNationalVersion, lastAirlineID)
		if err != nil || len(page) == 0 {
			return
		}
		for _, row := range page {
			model := models.MinPriceByAirline{
				FromID:          row.FromID,
				ToID:            row.ToID,
				NationalVersion: row.NationalVersion,
				AirlineID:       row.AirlineID,
				Currency:        row.Currency,
			}
			if price, ok := row.MinPrice.Get(); ok {
				model.MinPrice = ptr.Uint32(price)
				departureDate, _ := row.DepartureDate.Get()
				model.DepartureDate = ptr.String(departureDate)
			}
			if price, ok := row.MinPriceWithTransfers.Get(); ok {
				model.MinPriceWithTransfers = ptr.Uint32(price)
				departureDate, _ := row.DepartureDateWithTransfers.Get()
				model.DepartureDateWithTransfers = ptr.String(departureDate)
			}
			minPricesByAirloines = append(minPricesByAirloines, model)
		}
		lastFromID = minPricesByAirloines[len(minPricesByAirloines)-1].FromID
		lastToID = minPricesByAirloines[len(minPricesByAirloines)-1].ToID
		lastNationalVersion = minPricesByAirloines[len(minPricesByAirloines)-1].NationalVersion
		lastAirlineID = minPricesByAirloines[len(minPricesByAirloines)-1].AirlineID
	}
}

func (t *MinPricesByAirlineTable) getNextPage(
	limit int, lastFromID, lastToID uint32, lastNationalVersion string, lastAirlineID uint32,
) (page MinPriceByAirlineEntries, err error) {
	var query = fmt.Sprintf(
		`
			DECLARE $limit AS Uint64;
			DECLARE $last_from_id AS Uint32;
			DECLARE $last_to_id AS Uint32;
			DECLARE $last_national_version AS Utf8;
			DECLARE $last_airline_id AS Uint32;
	
			SELECT * FROM %[1]s
			WHERE (from_id, to_id, national_version, airline_id) > ($last_from_id, $last_to_id, $last_national_version, $last_airline_id)
			ORDER BY from_id, to_id, national_version, airline_id
			LIMIT $limit;
		`, t.tableName,
	)

	var res *table.Result

	err = table.Retry(
		context.Background(), t.sessionPool,
		table.OperationFunc(
			func(ctx context.Context, s *table.Session) (err error) {
				preparedStatement, err := s.Prepare(ctx, query)
				if err != nil {
					return err
				}
				_, res, err = preparedStatement.Execute(
					ctx, t.transactionControl,
					table.NewQueryParameters(
						table.ValueParam("$limit", ydb.Uint64Value(uint64(limit))),
						table.ValueParam("$last_from_id", ydb.Uint32Value(lastFromID)),
						table.ValueParam("$last_to_id", ydb.Uint32Value(lastToID)),
						table.ValueParam("$last_national_version", ydb.UTF8Value(lastNationalVersion)),
						table.ValueParam("$last_airline_id", ydb.Uint32Value(lastAirlineID)),
					),
				)
				return
			},
		),
	)

	if err != nil {
		return
	}
	if err = res.Err(); err != nil {
		return
	}

	if !res.NextSet() || !res.HasNextRow() {
		return
	}

	err = (&page).Scan(res)
	return
}
