package tables

import (
	"context"
	"fmt"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/landingcityto"
)

type CityToMonthAndYearPricesTable struct {
	sessionPool        *table.SessionPool
	transactionControl *table.TransactionControl
	tableName          string
	batchSize          uint64
}

func NewCityToMonthAndYearPricesTable(sessionPool *table.SessionPool, cfg Config) *CityToMonthAndYearPricesTable {
	return &CityToMonthAndYearPricesTable{
		sessionPool: sessionPool,
		transactionControl: table.TxControl(
			table.BeginTx(table.WithStaleReadOnly()),
			table.CommitTx(),
		),
		tableName: cfg.CityToMonthAndYearPricesTableName,
		batchSize: cfg.ReadTableBatchSize,
	}
}

func (t *CityToMonthAndYearPricesTable) GetAll() (cityToMonthAndYearPriceInfos []landingcityto.CityToMonthAndYearPriceInfo, err error) {
	lastToID := uint32(0)
	lastNationalVersion := ""
	var page CityToMonthAndYearPricesEntries
	for {
		page, err = t.getNextPage(
			t.batchSize,
			lastToID,
			lastNationalVersion,
		)
		if err != nil || len(page) == 0 {
			return
		}
		for _, row := range page {
			priceInfo := landingcityto.CityToMonthAndYearPriceInfo{
				ToID:                    row.ToID,
				NationalVersion:         row.NationalVersion,
				Currency:                row.Currency,
				YearMedianPrice:         row.YearMedianPrice,
				PopularMonthYear:        row.PopularMonthYear,
				PopularMonth:            row.PopularMonth,
				PopularMonthMedianPrice: row.PopularMonthMedianPrice,
				MinMonthYear:            row.MinMonthYear,
				MinMonth:                row.MinMonth,
				MinMonthMedianPrice:     row.MinMonthMedianPrice,
				MaxMonthYear:            row.MaxMonthYear,
				MaxMonth:                row.MaxMonth,
				MaxMonthMedianPrice:     row.MaxMonthMedianPrice,
			}
			cityToMonthAndYearPriceInfos = append(cityToMonthAndYearPriceInfos, priceInfo)
		}
		lastValue := cityToMonthAndYearPriceInfos[len(cityToMonthAndYearPriceInfos)-1]
		lastToID = lastValue.ToID
		lastNationalVersion = lastValue.NationalVersion
	}
}

func (t *CityToMonthAndYearPricesTable) getNextPage(
	limit uint64,
	lastToID uint32,
	lastNationalVersion string,
) (page CityToMonthAndYearPricesEntries, err error) {
	var query = fmt.Sprintf(
		`
			DECLARE $limit AS Uint64;
			DECLARE $last_to_id AS Uint32;
			DECLARE $last_national_version AS Utf8;

			SELECT * FROM %[1]s
			WHERE
				(to_id, national_version) > ($last_to_id, $last_national_version)
			ORDER BY to_id, national_version
			LIMIT $limit;
		`, t.tableName,
	)

	var res *table.Result

	err = table.Retry(
		context.Background(), t.sessionPool,
		table.OperationFunc(
			func(ctx context.Context, s *table.Session) (err error) {
				preparedStatement, err := s.Prepare(ctx, query)
				if err != nil {
					return err
				}
				_, res, err = preparedStatement.Execute(
					ctx, t.transactionControl,
					table.NewQueryParameters(
						table.ValueParam("$limit", ydb.Uint64Value(limit)),
						table.ValueParam("$last_to_id", ydb.Uint32Value(lastToID)),
						table.ValueParam("$last_national_version", ydb.UTF8Value(lastNationalVersion)),
					),
				)
				return
			},
		),
	)

	if err != nil {
		return
	}
	if err = res.Err(); err != nil {
		return
	}

	if !res.NextSet() || !res.HasNextRow() {
		return
	}

	err = (&page).Scan(res)
	return
}
