package tables

import (
	"context"
	"fmt"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type ReturnTicketPricesTable struct {
	sessionPool        *table.SessionPool
	transactionControl *table.TransactionControl
	tableName          string
	batchSize          int
}

func NewReturnTicketPricesTable(sessionPool *table.SessionPool, cfg Config) *ReturnTicketPricesTable {
	return &ReturnTicketPricesTable{
		sessionPool: sessionPool,
		transactionControl: table.TxControl(
			table.BeginTx(table.WithStaleReadOnly()),
			table.CommitTx(),
		),
		tableName: cfg.ReturnTicketPricesTableName,
		batchSize: 1000,
	}
}

func (t *ReturnTicketPricesTable) GetAll() (returnTicketPrices []models.ReturnTicketPrice, err error) {
	lastFromID := uint32(0)
	lastToID := uint32(0)
	lastNationalVersion := ""
	var page ReturnTicketPriceEntries
	for {
		page, err = t.getNextPage(t.batchSize, lastFromID, lastToID, lastNationalVersion)
		if err != nil || len(page) == 0 {
			return
		}
		for _, row := range page {
			returnTicketPrices = append(returnTicketPrices, models.ReturnTicketPrice(row))
		}
		lastFromID = returnTicketPrices[len(returnTicketPrices)-1].FromID
		lastToID = returnTicketPrices[len(returnTicketPrices)-1].ToID
		lastNationalVersion = returnTicketPrices[len(returnTicketPrices)-1].NationalVersion
	}
}

func (t *ReturnTicketPricesTable) getNextPage(
	limit int, lastFromID, lastToID uint32, lastNationalVersion string,
) (page ReturnTicketPriceEntries, err error) {
	var query = fmt.Sprintf(
		`
			DECLARE $limit AS Uint64;
			DECLARE $last_from_id AS Uint32;
			DECLARE $last_to_id AS Uint32;
			DECLARE $last_national_version AS Utf8;
	
			SELECT * FROM %[1]s
			WHERE (from_id, to_id, national_version) > ($last_from_id, $last_to_id, $last_national_version)
			ORDER BY from_id, to_id, national_version
			LIMIT $limit;
		`, t.tableName,
	)

	var res *table.Result

	err = table.Retry(
		context.Background(), t.sessionPool,
		table.OperationFunc(
			func(ctx context.Context, s *table.Session) (err error) {
				preparedStatement, err := s.Prepare(ctx, query)
				if err != nil {
					return err
				}
				_, res, err = preparedStatement.Execute(
					ctx, t.transactionControl,
					table.NewQueryParameters(
						table.ValueParam("$limit", ydb.Uint64Value(uint64(limit))),
						table.ValueParam("$last_from_id", ydb.Uint32Value(lastFromID)),
						table.ValueParam("$last_to_id", ydb.Uint32Value(lastToID)),
						table.ValueParam("$last_national_version", ydb.UTF8Value(lastNationalVersion)),
					),
				)
				return
			},
		),
	)

	if err != nil {
		return
	}
	if err = res.Err(); err != nil {
		return
	}

	if !res.NextSet() || !res.HasNextRow() {
		return
	}

	err = (&page).Scan(res)
	return
}
