package tables

import (
	"context"
	"fmt"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type RouteCrosslinksTable struct {
	sessionPool        *table.SessionPool
	transactionControl *table.TransactionControl
	tableName          string
	batchSize          int
}

func NewRouteCrosslinksTable(sessionPool *table.SessionPool, cfg Config) *RouteCrosslinksTable {
	return &RouteCrosslinksTable{
		sessionPool: sessionPool,
		transactionControl: table.TxControl(
			table.BeginTx(table.WithStaleReadOnly()),
			table.CommitTx(),
		),
		tableName: cfg.RouteCrosslinksTableName,
		// TODO (simon-ekb): Вынести batchSize в настройку. На 2021-07-08 в базе 443'000 записей,
		// поэтому чтение всей таблице в unstable по 1000 записей я не смог дождаться за час,
		// чтение по 10000 записей в unstable отработало за 34 минуты
		batchSize: 10000,
	}
}

func (t *RouteCrosslinksTable) GetAll() (routeCrosslinks []models.RouteCrosslink, err error) {
	lastFromID := uint32(0)
	lastToID := uint32(0)
	lastNationalVersion := ""
	lastPosition := uint8(0)
	lastCrosslinkFromID := uint32(0)
	lastCrosslinkToID := uint32(0)
	var page RouteCrosslinkEntries
	for {
		page, err = t.getNextPage(
			t.batchSize,
			lastFromID,
			lastToID,
			lastPosition,
			lastNationalVersion,
			lastCrosslinkFromID,
			lastCrosslinkToID,
		)
		if err != nil || len(page) == 0 {
			return
		}
		for _, row := range page {
			var price *uint32
			if p, ok := row.Price.Get(); ok {
				price = ptr.Uint32(p)
			}
			var currency *string
			if c, ok := row.Currency.Get(); ok {
				currency = ptr.String(c)
			}
			var date *string
			if d, ok := row.Date.Get(); ok {
				date = ptr.String(d)
			}
			crosslink := models.RouteCrosslink{
				FromID:          row.FromID,
				ToID:            row.ToID,
				NationalVersion: row.NationalVersion,
				CrosslinkFromID: row.CrosslinkFromID,
				CrosslinkToID:   row.CrosslinkToID,
				Price:           price,
				Currency:        currency,
				Date:            date,
			}
			lastPosition = row.Position
			routeCrosslinks = append(routeCrosslinks, crosslink)
		}
		lastFromID = routeCrosslinks[len(routeCrosslinks)-1].FromID
		lastToID = routeCrosslinks[len(routeCrosslinks)-1].ToID
		lastNationalVersion = routeCrosslinks[len(routeCrosslinks)-1].NationalVersion
		lastCrosslinkFromID = routeCrosslinks[len(routeCrosslinks)-1].CrosslinkFromID
		lastCrosslinkToID = routeCrosslinks[len(routeCrosslinks)-1].CrosslinkToID
	}
}

func (t *RouteCrosslinksTable) getNextPage(
	limit int,
	lastFromID, lastToID uint32,
	lastPosition uint8,
	lastNationalVersion string,
	lastCrosslinkFromID, lastCrosslinkToID uint32,
) (page RouteCrosslinkEntries, err error) {
	var query = fmt.Sprintf(
		`
			DECLARE $limit AS Uint64;
			DECLARE $last_from_id AS Uint32;
			DECLARE $last_to_id AS Uint32;
			DECLARE $last_national_version AS Utf8;
			DECLARE $last_position AS Uint8;
			DECLARE $last_crosslink_from_id AS Uint32;
			DECLARE $last_crosslink_to_id AS Uint32;

			SELECT * FROM %[1]s
			WHERE
				(from_id, to_id, national_version, position, crosslink_from_id, crosslink_to_id) > ($last_from_id, $last_to_id, $last_national_version, $last_position, $last_crosslink_from_id, $last_crosslink_to_id)
			ORDER BY from_id, to_id, national_version, position, crosslink_from_id, crosslink_to_id
			LIMIT $limit;
		`, t.tableName,
	)

	var res *table.Result

	err = table.Retry(
		context.Background(), t.sessionPool,
		table.OperationFunc(
			func(ctx context.Context, s *table.Session) (err error) {
				preparedStatement, err := s.Prepare(ctx, query)
				if err != nil {
					return err
				}
				_, res, err = preparedStatement.Execute(
					ctx, t.transactionControl,
					table.NewQueryParameters(
						table.ValueParam("$limit", ydb.Uint64Value(uint64(limit))),
						table.ValueParam("$last_from_id", ydb.Uint32Value(lastFromID)),
						table.ValueParam("$last_to_id", ydb.Uint32Value(lastToID)),
						table.ValueParam("$last_national_version", ydb.UTF8Value(lastNationalVersion)),
						table.ValueParam("$last_position", ydb.Uint8Value(lastPosition)),
						table.ValueParam("$last_crosslink_from_id", ydb.Uint32Value(lastCrosslinkFromID)),
						table.ValueParam("$last_crosslink_to_id", ydb.Uint32Value(lastCrosslinkToID)),
					),
				)
				return
			},
		),
	)

	if err != nil {
		return
	}
	if err = res.Err(); err != nil {
		return
	}

	if !res.NextSet() || !res.HasNextRow() {
		return
	}

	err = (&page).Scan(res)
	return
}
