package tables

import (
	"context"
	"encoding/json"
	"fmt"
	"strings"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type RouteInfosTable struct {
	sessionPool        *table.SessionPool
	transactionControl *table.TransactionControl
	tableName          string
	batchSize          int
}

func NewRouteInfosTable(sessionPool *table.SessionPool, cfg Config) *RouteInfosTable {
	return &RouteInfosTable{
		sessionPool: sessionPool,
		transactionControl: table.TxControl(
			table.BeginTx(table.WithStaleReadOnly()),
			table.CommitTx(),
		),
		tableName: cfg.RouteInfosTableName,
		batchSize: 1000,
	}
}

func (t *RouteInfosTable) GetAll() (routeInfos []models.RouteInfo, err error) {
	lastFromID := uint32(0)
	lastToID := uint32(0)
	var page RouteInfoEntries
	for {
		page, err = t.getNextPage(t.batchSize, lastFromID, lastToID)
		if err != nil || len(page) == 0 {
			return
		}
		for _, row := range page {
			fromAirports := make([]int, 0)
			if err = json.NewDecoder(strings.NewReader(row.FromAirports)).Decode(&fromAirports); err != nil {
				return nil, err
			}
			toAirports := make([]int, 0)
			if err = json.NewDecoder(strings.NewReader(row.ToAirports)).Decode(&toAirports); err != nil {
				return nil, err
			}
			routeInfos = append(
				routeInfos,
				models.RouteInfo{
					FromID:       row.FromID,
					ToID:         row.ToID,
					Distance:     row.Distance,
					Duration:     row.Duration,
					FromAirports: fromAirports,
					ToAirports:   toAirports,
				},
			)
		}
		lastFromID = routeInfos[len(routeInfos)-1].FromID
		lastToID = routeInfos[len(routeInfos)-1].ToID
	}
}

func (t *RouteInfosTable) getNextPage(
	limit int, lastFromID, lastToID uint32,
) (page RouteInfoEntries, err error) {
	var query = fmt.Sprintf(
		`
			DECLARE $limit AS Uint64;
			DECLARE $last_from_id AS Uint32;
			DECLARE $last_to_id AS Uint32;

			SELECT * FROM %[1]s
			WHERE (from_id, to_id) > ($last_from_id, $last_to_id)
			ORDER BY from_id, to_id
			LIMIT $limit;
		`, t.tableName,
	)

	var res *table.Result

	err = table.Retry(
		context.Background(), t.sessionPool,
		table.OperationFunc(
			func(ctx context.Context, s *table.Session) (err error) {
				preparedStatement, err := s.Prepare(ctx, query)
				if err != nil {
					return err
				}
				_, res, err = preparedStatement.Execute(
					ctx, t.transactionControl,
					table.NewQueryParameters(
						table.ValueParam("$limit", ydb.Uint64Value(uint64(limit))),
						table.ValueParam("$last_from_id", ydb.Uint32Value(lastFromID)),
						table.ValueParam("$last_to_id", ydb.Uint32Value(lastToID)),
					),
				)
				return
			},
		),
	)

	if err != nil {
		return
	}
	if err = res.Err(); err != nil {
		return
	}

	if !res.NextSet() || !res.HasNextRow() {
		return
	}

	err = (&page).Scan(res)
	return
}
