package tables

import (
	"context"
	"fmt"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type RoutesTable struct {
	sessionPool        *table.SessionPool
	transactionControl *table.TransactionControl
	tableName          string
	batchSize          int
}

func NewRoutesTable(sessionPool *table.SessionPool, cfg Config) *RoutesTable {
	return &RoutesTable{
		sessionPool: sessionPool,
		transactionControl: table.TxControl(
			table.BeginTx(table.WithStaleReadOnly()),
			table.CommitTx(),
		),
		tableName: cfg.RoutesTableName,
		batchSize: 1000,
	}
}

func (t *RoutesTable) GetAll() (routes []models.Route, err error) {
	lastFromID := uint32(0)
	lastToID := uint32(0)
	lastNationalVersion := ""
	var page RouteEntries
	for {
		page, err = t.getNextPage(t.batchSize, lastFromID, lastToID, lastNationalVersion)
		if err != nil || len(page) == 0 {
			return
		}
		for _, row := range page {
			routes = append(routes, models.Route(row))
		}
		lastFromID = routes[len(routes)-1].FromID
		lastToID = routes[len(routes)-1].ToID
		lastNationalVersion = routes[len(routes)-1].NationalVersion
	}
}

func (t *RoutesTable) getNextPage(
	limit int, lastFromID, lastToID uint32, lastNationalVersion string,
) (page RouteEntries, err error) {
	var query = fmt.Sprintf(
		`
			DECLARE $limit AS Uint64;
			DECLARE $last_from_id AS Uint32;
			DECLARE $last_to_id AS Uint32;
			DECLARE $last_national_version AS Utf8;

			SELECT * FROM %[1]s
			WHERE (from_id, to_id, national_version) > ($last_from_id, $last_to_id, $last_national_version)
			ORDER BY from_id, to_id, national_version
			LIMIT $limit;
		`, t.tableName,
	)

	var res *table.Result

	err = table.Retry(
		context.Background(), t.sessionPool,
		table.OperationFunc(
			func(ctx context.Context, s *table.Session) (err error) {
				preparedStatement, err := s.Prepare(ctx, query)
				if err != nil {
					return err
				}
				_, res, err = preparedStatement.Execute(
					ctx, t.transactionControl,
					table.NewQueryParameters(
						table.ValueParam("$limit", ydb.Uint64Value(uint64(limit))),
						table.ValueParam("$last_from_id", ydb.Uint32Value(lastFromID)),
						table.ValueParam("$last_to_id", ydb.Uint32Value(lastToID)),
						table.ValueParam("$last_national_version", ydb.UTF8Value(lastNationalVersion)),
					),
				)
				return
			},
		),
	)

	if err != nil {
		return
	}
	if err = res.Err(); err != nil {
		return
	}

	if !res.NextSet() || !res.HasNextRow() {
		return
	}

	err = (&page).Scan(res)
	return
}
