package handler

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"time"

	"github.com/go-chi/chi/v5"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/service/landings"
	"a.yandex-team.ru/travel/library/go/httputil"
	"a.yandex-team.ru/travel/library/go/metrics"
)

type httpResponse struct {
	Result *models.Landing     `json:"result"`
	Error  *httputil.HTTPError `json:"error,omitempty"`
}

type HTTPLandingHandler struct {
	landingBuilder LandingBuilder
	logger         log.Logger
}

func NewHTTPLandingHandler(landingBuilder LandingBuilder, logger log.Logger) *HTTPLandingHandler {
	return &HTTPLandingHandler{landingBuilder: landingBuilder, logger: logger}
}

func (h *HTTPLandingHandler) GetLanding(w http.ResponseWriter, r *http.Request) {
	start := time.Now()
	span, ctx := opentracing.StartSpanFromContext(r.Context(), "internal.handler.HTTPLandingHandler: GetLanding")
	defer span.Finish()

	landingQuery, err := h.parseQuery(r)
	if err != nil {
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}
	defer h.writeTimings(start)
	h.logger.Infof("GetLanding request %+v", landingQuery)
	landing, err := h.landingBuilder.Build(ctx, landingQuery)
	if err != nil {
		if errors.Is(err, models.ErrorNotFound) {
			metrics.GlobalAppMetrics().GetOrCreateCounter(metricsPrefix, nil, "not_found").Inc()
			httputil.HandleError(fmt.Errorf("not found"), http.StatusNotFound, w)
			return
		} else {
			metrics.GlobalAppMetrics().GetOrCreateCounter(metricsPrefix, nil, "error").Inc()
			h.logger.Error("internal error", log.Error(err))
			httputil.HandleError(fmt.Errorf("internal error"), http.StatusInternalServerError, w)
			return
		}
	}

	result := httpResponse{Result: landing}
	resultBytes, err := json.Marshal(result)
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}
	w.WriteHeader(http.StatusOK)
	w.Header().Set("Content-Type", "application/json")
	_, _ = w.Write(resultBytes)
}

func (h *HTTPLandingHandler) GetRouteBuilder() func(r chi.Router) {
	return func(r chi.Router) {
		r.Get("/landing/route/", h.GetLanding)
	}
}

func (h *HTTPLandingHandler) parseQuery(r *http.Request) (*landings.Query, error) {
	params, err := url.ParseQuery(r.URL.RawQuery)
	if err != nil {
		return nil, err
	}

	rawFromID := params.Get("fromId")
	if rawFromID == "" {
		return nil, fmt.Errorf("missing required param 'fromId'")
	}
	fromID64, err := strconv.ParseUint(rawFromID, 10, 32)
	if err != nil {
		return nil, fmt.Errorf("incorrect param 'from_id': expected positive integer but got %v", rawFromID)
	}
	fromID := uint32(fromID64)

	rawToID := params.Get("toId")
	if rawToID == "" {
		return nil, fmt.Errorf("missing required param 'toId'")
	}
	toID64, err := strconv.ParseUint(rawToID, 10, 32)
	if err != nil {
		return nil, fmt.Errorf("incorrect param 'toId': expected positive integer but got %v", rawToID)
	}
	toID := uint32(toID64)

	nationalVersion := params.Get("nationalVersion")
	if nationalVersion == "" {
		return nil, fmt.Errorf("missing required param 'nationalVersion'")
	}

	rawFields := params.Get("fields")
	var fields []string
	if rawFields != "" {
		fields = strings.Split(rawFields, ",")
	}

	return landings.NewQuery(fromID, toID, nationalVersion, fields), nil
}

func (h *HTTPLandingHandler) writeTimings(start time.Time) {
	metrics.GlobalAppMetrics().GetOrCreateHistogram(
		metricsPrefix,
		nil,
		"timings",
		timingBuckets,
	).RecordDuration(time.Since(start))
}
