package landings

import (
	"context"
	"fmt"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type AlternativeRoutesPricesGetter func(context.Context, uint32, uint32, string) ([]models.AlternativeRoutePrice, error)
type MedianPricesGetter func(context.Context, uint32, uint32, string) (*models.MedianPrices, error)
type PopularMonthGetter func(context.Context, uint32, uint32, string) (*models.PopularMonth, error)
type ReturnTicketPriceGetter func(context.Context, uint32, uint32, string) (*models.ReturnTicketPrice, error)
type RouteInfoGetter func(context.Context, uint32, uint32) (*models.RouteInfo, error)
type TopAirlinesGetter func(context.Context, uint32, uint32, string) ([]uint32, error)
type MinPricesByAirlineGetter func(context.Context, uint32, uint32, string, uint32) (*models.MinPriceByAirline, error)
type RouteCrosslinksGetter func(context.Context, uint32, uint32, string) ([]models.RouteCrosslink, error)
type RouteValidator func(context.Context, models.Route) bool

type landingFieldBuilder func(ctx context.Context, landing *models.Landing, query *Query) error

type Service struct {
	logger         log.Logger
	fieldsBuilders map[string]landingFieldBuilder

	alternativeRoutesPricesGetter AlternativeRoutesPricesGetter
	medianPricesGetter            MedianPricesGetter
	popularMonthGetter            PopularMonthGetter
	returnTicketPriceGetter       ReturnTicketPriceGetter
	routeInfoGetter               RouteInfoGetter
	topAirlinesGetter             TopAirlinesGetter
	minPricesByAirlineGetter      MinPricesByAirlineGetter
	routeCrosslinksGetter         RouteCrosslinksGetter
	routeValidator                RouteValidator
}

type Query struct {
	FromID          uint32
	ToID            uint32
	NationalVersion string
	Fields          []string
}

func NewQuery(fromID uint32, toID uint32, nationalVersion string, fields []string) *Query {
	return &Query{FromID: fromID, ToID: toID, NationalVersion: nationalVersion, Fields: fields}
}

func (s *Service) Build(ctx context.Context, query *Query) (
	result *models.Landing, err error,
) {
	span, ctx := opentracing.StartSpanFromContext(ctx, "internal.service.landings.Service:Build")
	defer span.Finish()

	route := models.Route{FromID: query.FromID, ToID: query.ToID, NationalVersion: query.NationalVersion}
	if !s.routeValidator(ctx, route) {
		return nil, models.ErrorNotFound
	}

	landing := &models.Landing{FromID: query.FromID, ToID: query.ToID}
	emptyLanding := true
	if len(query.Fields) > 0 {
		for _, field := range query.Fields {
			if builder, ok := s.fieldsBuilders[field]; ok {
				err = builder(ctx, landing, query)
				emptyLanding = emptyLanding && err != nil
			}
		}
	} else {
		for field := range s.fieldsBuilders {
			err = s.fieldsBuilders[field](ctx, landing, query)
			emptyLanding = emptyLanding && err != nil
		}
	}
	if emptyLanding {
		return nil, models.ErrorNotFound
	}
	return landing, nil
}

func (s *Service) buildAlternativeRoutesPrices(ctx context.Context, landing *models.Landing, query *Query) error {
	if alternativeRoutesPrices, err := s.alternativeRoutesPricesGetter(ctx, query.FromID, query.ToID, query.NationalVersion); err != nil {
		return fmt.Errorf("no alternative routes prices for query: %+v", query)
	} else {
		landing.AlternativeRoutesPrices = alternativeRoutesPrices
	}
	return nil
}

func (s *Service) buildRouteCrosslinks(ctx context.Context, landing *models.Landing, query *Query) error {
	if routeCrosslinks, err := s.routeCrosslinksGetter(ctx, query.FromID, query.ToID, query.NationalVersion); err != nil {
		return fmt.Errorf("no route crosslinks for query: %+v", query)
	} else {
		landing.RouteCrosslinks = routeCrosslinks
	}
	return nil
}

func (s *Service) buildMedianPrices(ctx context.Context, landing *models.Landing, query *Query) error {
	if medianPrices, err := s.medianPricesGetter(ctx, query.FromID, query.ToID, query.NationalVersion); err != nil {
		return fmt.Errorf("no median prices for query: %+v", query)
	} else {
		landing.MedianPrices = medianPrices
	}
	return nil
}

func (s *Service) buildPopularMonth(ctx context.Context, landing *models.Landing, query *Query) error {
	if popularMonth, err := s.popularMonthGetter(ctx, query.FromID, query.ToID, query.NationalVersion); err != nil {
		return fmt.Errorf("no popular month for query: %+v", query)
	} else {
		landing.PopularMonth = popularMonth
	}
	return nil
}

func (s *Service) buildReturnTicketPrice(ctx context.Context, landing *models.Landing, query *Query) error {
	if returnTicketPrice, err := s.returnTicketPriceGetter(ctx, query.FromID, query.ToID, query.NationalVersion); err != nil {
		return fmt.Errorf("no return ticket price for query: %+v", query)
	} else {
		landing.ReturnTicketPrice = returnTicketPrice
	}
	return nil
}

func (s *Service) buildRouteInfo(ctx context.Context, landing *models.Landing, query *Query) error {
	if routeInfo, err := s.routeInfoGetter(ctx, query.FromID, query.ToID); err != nil {
		return fmt.Errorf("no route info for query: %+v", query)
	} else {
		landing.RouteInfo = routeInfo
	}
	return nil
}

func (s *Service) buildTopAirlines(ctx context.Context, landing *models.Landing, query *Query) error {
	topAirlines, err := s.topAirlinesGetter(ctx, query.FromID, query.ToID, query.NationalVersion)
	if err != nil {
		return fmt.Errorf("no top airlines for query: %+v", query)
	}
	landing.TopAirlines = make([]models.AirlineWithPrice, 0)
	for _, airlineID := range topAirlines {
		price, err := s.minPricesByAirlineGetter(ctx, query.FromID, query.ToID, query.NationalVersion, airlineID)
		if err == models.ErrorNotFound || price == nil {
			continue
		}
		airlineWithPrice := models.AirlineWithPrice{
			ID:                         airlineID,
			Currency:                   price.Currency,
			Price:                      price.MinPrice,
			DepartureDate:              price.DepartureDate,
			PriceWithTransfers:         price.MinPriceWithTransfers,
			DepartureDateWithTransfers: price.DepartureDateWithTransfers,
		}
		landing.TopAirlines = append(landing.TopAirlines, airlineWithPrice)
	}
	return nil
}

func NewService(
	l log.Logger,
	alternativeRoutesPricesGetter AlternativeRoutesPricesGetter,
	medianPricesGetter MedianPricesGetter,
	popularMonthsGetter PopularMonthGetter,
	returnTicketPriceGetter ReturnTicketPriceGetter,
	routeInfoGetter RouteInfoGetter,
	topAirlinesGetter TopAirlinesGetter,
	minPricesByAirlineGetter MinPricesByAirlineGetter,
	routeCrosslinksGetter RouteCrosslinksGetter,
	routeValidator RouteValidator,
) *Service {
	service := &Service{
		logger: l,

		alternativeRoutesPricesGetter: alternativeRoutesPricesGetter,
		medianPricesGetter:            medianPricesGetter,
		popularMonthGetter:            popularMonthsGetter,
		returnTicketPriceGetter:       returnTicketPriceGetter,
		routeInfoGetter:               routeInfoGetter,
		topAirlinesGetter:             topAirlinesGetter,
		minPricesByAirlineGetter:      minPricesByAirlineGetter,
		routeCrosslinksGetter:         routeCrosslinksGetter,
		routeValidator:                routeValidator,
	}
	service.fieldsBuilders = map[string]landingFieldBuilder{
		alternativeRoutesPricesField: service.buildAlternativeRoutesPrices,
		medianPricesField:            service.buildMedianPrices,
		popularMonthField:            service.buildPopularMonth,
		returnTicketPriceField:       service.buildReturnTicketPrice,
		routeInfoField:               service.buildRouteInfo,
		topAirlinesField:             service.buildTopAirlines,
		routeCrosslinksField:         service.buildRouteCrosslinks,
	}
	return service
}

const (
	alternativeRoutesPricesField string = "alternativeRoutesPrices"
	medianPricesField            string = "medianPrices"
	popularMonthField            string = "popularMonth"
	returnTicketPriceField       string = "returnTicketPrice"
	routeInfoField               string = "routeInfo"
	topAirlinesField             string = "topAirlines"
	routeCrosslinksField         string = "routeCrosslinks"
)
