# -*- coding: utf-8 -*-
import logging
from datetime import datetime, date, timedelta
from argparse import ArgumentParser

import ydb

from travel.avia.library.python.lib_yt.client import configured_client
from travel.avia.library.python.references.station import create_station_cache
from travel.avia.library.python.references.station_to_settlement import create_station_to_settlement_cache
from travel.avia.avia_statistics.flights_updater.lib.updater import FlightsUpdater
from travel.avia.avia_statistics.flights_updater.lib.collector import FlightsCollector
from travel.avia.avia_statistics.flights_updater.lib.table import FlightsTable
from travel.avia.library.python.ydb.session_manager import YdbSessionManager


def main():
    parser = ArgumentParser()

    parser.add_argument('--debug', action='store_true', default=False)
    parser.add_argument('--shared-flights-api-host', help='shared flights api base url for iata correction',
                        default='http://shared-flights.production.avia.yandex.net/api/v1')

    parser.add_argument('--yt-token', help='YT token', required=True)
    parser.add_argument('--yt-proxy', default='hahn')

    parser.add_argument('--ydb-cluster', default='ydb-ru-prestable.yandex.net:2135')
    parser.add_argument('--ydb-database', default='/ru-prestable/ticket/testing/avia_statistics')
    parser.add_argument('--ydb-table', default='flights')
    parser.add_argument('--ydb-token', help='YDB token', required=True)

    parser.add_argument('--batch', help='Batch size', default=5000)

    parser.add_argument('--variants-log-directory', help='Avia variants log directory',
                        default='//home/logfeller/logs/avia-variants-log/1d')
    parser.add_argument('--date', help='Date', default=(date.today() - timedelta(days=1)).isoformat())

    arguments = parser.parse_args()

    logging.basicConfig(level=logging.INFO)

    if arguments.debug:
        logging.basicConfig(level=logging.DEBUG)

    yt_client = configured_client(proxy=arguments.yt_proxy, token=arguments.yt_token)
    station_cache = create_station_cache(yt_client)
    station_to_settlement_cache = create_station_to_settlement_cache(yt_client)
    collector = FlightsCollector(
        yt_client,
        arguments.variants_log_directory,
        station_cache,
        station_to_settlement_cache,
    )
    ydb_config = ydb.DriverConfig(
        endpoint=arguments.ydb_cluster,
        database=arguments.ydb_database,
        auth_token=arguments.ydb_token,
    )

    session_manager = YdbSessionManager(ydb_config)
    flights_table = FlightsTable(session_manager, arguments.ydb_database, arguments.ydb_table)
    flights_updater = FlightsUpdater(collector, arguments.shared_flights_api_host, flights_table, arguments.batch)
    target_date = datetime.strptime(arguments.date, '%Y-%m-%d')
    flights_updater.update_flights(target_date)


if __name__ == '__main__':
    main()
