from collections import namedtuple, defaultdict
from typing import Set, Dict, Generator, List

from yt.wrapper import YtClient

LandingRoute = namedtuple('LandingRoute', ('from_id', 'to_id', 'national_version',))
RouteWeight = namedtuple('RouteWeight', ('redirs', 'searches',))


def get_landing_routes(yt_client, path='//home/avia/avia-statistics/landing-routes'):
    # type: (YtClient, str) -> Set[LandingRoute]
    return set(iter_landing_routes(yt_client, path))


def iter_landing_routes(yt_client, path='//home/avia/avia-statistics/landing-routes'):
    # type: (YtClient, str) -> Generator[LandingRoute]
    routes = yt_client.read_table(path)
    for r in routes:
        yield LandingRoute(r['from_id'], r['to_id'], r['national_version'])


def get_route_weights(yt_client, path='//home/avia/avia-statistics/landing-route-weights'):
    # type: (YtClient, str) -> Dict[LandingRoute, RouteWeight]
    return {
        LandingRoute(r['from_id'], r['to_id'], r['national_version']): RouteWeight(r['redirs'] or 0, r['searches'] or 0)
        for r in yt_client.read_table(path)
    }


def get_landing_route_crosslinks(yt_client, path='//home/avia/avia-statistics/landing-route-crosslinks'):
    # type: (YtClient, str) -> Dict[LandingRoute, List[LandingRoute]]
    crosslinks = defaultdict(list)
    for r in yt_client.read_table(path):
        route = LandingRoute(r['from_id'], r['to_id'], r['national_version'])
        crosslinks[route].append(LandingRoute(r['crosslink_from_id'], r['crosslink_to_id'], r['national_version']))
    return crosslinks
