# -*- coding: utf-8 -*-
from operator import itemgetter
from typing import List, Generator, Tuple, Optional

import six
from geoindex import GeoGridIndex, GeoPoint

from travel.library.python.dicts.avia.direction_national_repository import DirectionNationalRepository
from travel.library.python.dicts.avia.settlement_repository import SettlementRepository
from travel.proto.dicts.avia.settlements_pb2 import TSettlement


class SettlementsGeoIndex(object):
    def __init__(self, direction_national_repository, settlement_repository):
        # type: (DirectionNationalRepository, SettlementRepository) -> None

        self._settlement_repository = settlement_repository
        self._geo_index = GeoGridIndex(precision=2)
        settlement_ids = {
            s.Id: s for s in settlement_repository.itervalues()
            if not s.Hidden and 'plane' in s.TypeChoices
        }
        arrival_settlement_ids = {d.ArrivalSettlementID for d in direction_national_repository.itervalues()}

        for settlement_id, settlement in six.iteritems(settlement_ids):
            if (
                settlement.Latitude and settlement.Longitude
                and settlement_id in arrival_settlement_ids
            ):
                self._geo_index.add_point(self._settlement_to_geo_point(settlement))

    @staticmethod
    def _settlement_to_geo_point(settlement):
        return GeoPoint(settlement.Latitude, settlement.Longitude, ref=settlement.Id)

    def get_distance(self, from_id, to_id):
        # type: (int, int) -> Optional[int]
        from_settlement = self._settlement_repository.get(from_id)
        to_settlement = self._settlement_repository.get(to_id)
        if not (from_settlement.Latitude or from_settlement.Longitude) or \
                not (to_settlement.Latitude or to_settlement.Longitude):
            return None
        return int(self._settlement_to_geo_point(from_settlement).distance_to(
            self._settlement_to_geo_point(to_settlement))
        )

    def get_nearest_with_distance(self, settlement_id, distance):
        # type: (int, int) -> Generator[Tuple[TSettlement, int]]
        settlement = self._settlement_repository.get(settlement_id)
        if not settlement:
            return
        center = self._settlement_to_geo_point(settlement)
        nearest = self._geo_index.get_nearest_points(center, distance, 'km')
        for point, distance in sorted(nearest, key=itemgetter(1)):
            if point.ref == settlement_id:
                continue
            yield self._settlement_repository.get(point.ref), distance

    def get_nearest(self, settlement_id, distance):
        # type: (int, int) -> List[TSettlement]
        return map(
            itemgetter(0),
            self.get_nearest_with_distance(settlement_id, distance)
        )
