# -*- coding: utf-8 -*-
from datetime import date
from dateutil.relativedelta import relativedelta
from typing import Generator, Tuple
import logging

from yql.api.v1.client import YqlClient  # noqa

from travel.avia.library.python.references.station_to_settlement import StationToSettlementCache  # noqa
from travel.avia.library.python.references.station import StationCache  # noqa
from travel.avia.avia_statistics.lib.consts import CURRENCY_BY_NATIONAL_VERSION

logger = logging.getLogger(__name__)

MEDIAN_PRICES_QUERY_TEMPLATE = """
use hahn;
pragma yson.DisableStrict;
pragma yt.InferSchema;

$logs_date_from = '{logs_date_from}';
$logs_date_to = '{logs_date_to}';

$point_key_to_id = ($pk) -> {{
    return CAST(SUBSTRING($pk, 1) as Uint32)
}};

$parse = DateTime::Parse("%Y-%m-%d");

$fmt = DateTime::Format("%Y-%m-%d");

$current_month_first_day_year_ago = $fmt(DateTime::MakeDate(DateTime::StartOfMonth(DateTime::MakeDate(DateTime::Update(
    CurrentUtcDate(),
    DateTime::GetYear(CurrentUtcDate()) - 1us AS Year
)))));

$previous_month_last_day = $fmt(DateTime::MakeDate(DateTime::StartOfMonth(CurrentUtcDate())) - DateTime::IntervalFromDays(1));

$currency_by_nv = AsDict({currency_by_nv});

$redirs = (
    select
        $point_key_to_id(FROMID) as from_id,
        $point_key_to_id(TOID) as to_id,
        NATIONAL_VERSION as national_version,
        `WHEN` as forward_date,
        ORIGINAL_PRICE as price,
        ORIGINAL_CURRENCY as currency,
        DateTime::GetYear($parse(`WHEN`)) as year,
        DateTime::GetMonth($parse(`WHEN`)) as month
    from range(`{redirects_log_path}`,  $logs_date_from, $logs_date_to)
    WHERE
        `FILTER` = 0
        and KLASS = 'economy'
        and (`WHEN` between $current_month_first_day_year_ago and $previous_month_last_day)
        and DictContains($currency_by_nv, NATIONAL_VERSION)
        and $currency_by_nv[NATIONAL_VERSION] = ORIGINAL_CURRENCY
);

$year_median_by_route = (
    select
        from_id,
        to_id,
        national_version,
        CAST(median(price) as Uint32) as year_median_price,
        MIN(currency) as currency
    from $redirs
    group by from_id, to_id, national_version
);

$month_medians_by_route = (
    select
        from_id,
        to_id,
        national_version,
        year,
        month,
        CAST(median(price) as Uint32) as month_median_price,
        MIN(currency) as currency,
        COUNT(*) as redirs_count
    from $redirs
    group by from_id, to_id, national_version, year, month
);

$min_by_month = (
    select
        from_id,
        to_id,
        national_version,
        MIN(AsTuple(month_median_price, year, month)).0 as month_median_price,
        MIN(AsTuple(month_median_price, year, month)).1 as year,
        MIN(AsTuple(month_median_price, year, month)).2 as month,
        MIN(currency) as currency
    from $month_medians_by_route
    group by from_id, to_id, national_version
);

select
    y.from_id as from_id,
    y.to_id as to_id,
    y.national_version as national_version,
    y.year_median_price as year_median_price,
    m.month_median_price as month_median_price,
    m.year as year,
    m.month as month,
    m.currency as currency,
from $year_median_by_route as y
join $min_by_month as m
on y.from_id = m.from_id and y.to_id = m.to_id and y.national_version = m.national_version
join hahn.`//home/avia/avia-statistics/landing-routes` as r
on y.from_id = r.from_id and y.to_id = r.to_id and y.national_version = r.national_version
"""


class RunYqlException(Exception):
    pass


class MedianPricesCollector(object):
    def __init__(
        self,
        yql_client,
        redirects_log_path,
        station_cache,
        station_to_settlement_cache,
    ):
        """
        :param YqlClient yql_client:
        :param str redirects_log_path:
        :param StationCache station_cache:
        :param StationToSettlementCache station_to_settlement_cache:
        """
        self._yql_client = yql_client
        self._redirects_log_path = redirects_log_path
        self._station_cache = station_cache
        self._station_to_settlement = station_to_settlement_cache

    def collect_median_prices(self, today):
        # type: (date) -> Generator[Tuple]
        date_from = (today - relativedelta(years=1, months=6, day=1)).strftime('%Y-%m-%d')  # 1.5 year ago
        date_to = (today - relativedelta(days=today.day)).strftime('%Y-%m-%d')  # previous month last day
        logger.info(
            'calculating median prices in range (%s) - (%s) in %s',
            date_from,
            date_to,
            self._redirects_log_path,
        )
        query = MEDIAN_PRICES_QUERY_TEMPLATE.format(
            logs_date_from=date_from,
            logs_date_to=date_to,
            redirects_log_path=self._redirects_log_path,
            currency_by_nv=', '.join(
                'AsTuple(\'{}\', \'{}\')'.format(k, v)
                for k, v in CURRENCY_BY_NATIONAL_VERSION.items()
            ),
        )

        yql_query = self._yql_client.query(query, syntax_version=1)
        yql_query.run()
        yql_query.get_results(wait=True)

        if not yql_query.is_ok:
            if yql_query.errors:
                for error in yql_query.errors:
                    logger.error(' - %s', str(error))
            raise RunYqlException('Error while running YQL')

        logger.info('YQL query is done')

        yql_query.table.fetch_full_data()
        for row in yql_query.table.rows:
            yield row
