# -*- coding: utf-8 -*-
import logging
from datetime import date, timedelta

from yql.api.v1.client import YqlClient

from travel.library.python.dicts.avia.company_repository import CompanyRepository

logger = logging.getLogger(__name__)

MIN_PRICE_BY_AIRLINE_YQL_QUERY_TEMPLATE = """
use hahn;
pragma yson.DisableStrict;

$logs_from_date = '{logs_from_date}';
$logs_to_date = '{logs_to_date}';


$get_id = ($settlement_id, $id) -> {{
    return IF(
        $settlement_id is null,
        $id,
        'c' || CAST($settlement_id as String)
    );
}};

$redirs = (
    select
        `MARKER` as marker,
        `ORIGINAL_PRICE` as price,
        `ORIGINAL_CURRENCY` as currency,
        `VARIANTID` as variant_id,
        `NATIONAL_VERSION` as national_version,
        `WHEN` as forward_date,
        $get_id(Yson::LookupInt64(`_REST`, 'settlement_from_id'), FROMID) as from_id,
        $get_id(Yson::LookupInt64(`_REST`, 'settlement_to_id'), TOID) as to_id
    from range('//home/avia/logs/avia-redir-balance-by-day-log', $logs_from_date, $logs_to_date)
    where `FILTER` = 0 AND `RETURN_DATE` = ""
);

$actions = (
    select
        variantId as variant_id,
        forward,
        backward
    from range('//home/logfeller/logs/avia-action-show-log/1d', $logs_from_date, $logs_to_date)
);

$GetRoute = ($routeStr) -> {{
    RETURN IF(
        $routeStr != "",
        ListMap(
            String::SplitToList($routeStr, ","),
            ($x) -> {{
                RETURN COALESCE(String::SplitToList($x, ".")[0], "");
            }}
        ),
        CAST([] as List<String>),
    )
}};

$books_with_segments = (
    select
        from_id,
        to_id,
        national_version,
        price,
        currency,
        forward_date,
        $GetRoute(forward) as forward_segments,
        forward as forward_segments_key,
    from (
        select *
        from $redirs
        where String::StartsWith(from_id, 'c') and String::StartsWith(to_id, 'c')
    ) as r
    join $actions as a on r.variant_id = a.variant_id
);


$get_company_code = ($segments) -> {{
    return Unicode::Substring(CAST($segments[0] as Utf8), 0, 2);
}};

$single_company_segments = ($segments) -> {{
    return ListLength(ListUniq(ListMap($segments, ($s) -> {{ return Unicode::Substring(CAST($s as Utf8), 0, 2); }}))) = 1;
}};

$departure_date_from = CAST(DateTime::MakeDate(CurrentTzDatetime('Europe/Moscow')) + Datetime::IntervalFromDays(2) as String);
$departure_date_to = CAST(DateTime::MakeDate(CurrentTzDatetime('Europe/Moscow')) + Datetime::IntervalFromDays(31) as String);

$books_with_segments_on_date = (
    select
        from_id,
        to_id,
        national_version,
        price,
        currency,
        forward_date as departure_date,
        ListLength(forward_segments) > 1 as with_transfers,
        $get_company_code(forward_segments) as company_code,
    from $books_with_segments
    where $single_company_segments(forward_segments) and forward_date >= $departure_date_from and forward_date <= $departure_date_to
);

$min_price_by_company = (
    select
        from_id,
        to_id,
        national_version,
        company_code,
        with_transfers,
        MIN(AsTuple(price, departure_date)).1 as departure_date,
        CAST(MIN(price) as Uint32) as min_price,
        currency,
    from $books_with_segments_on_date
    group by from_id, to_id, national_version, company_code, currency, with_transfers
);


select
    from_id,
    to_id,
    national_version,
    company_code,
    MIN(IF(not with_transfers, min_price)) as min_price,
    MIN(IF(not with_transfers, departure_date)) as departure_date,
    MIN(IF(with_transfers, min_price)) as min_price_with_transfers,
    MIN(IF(with_transfers, departure_date)) as departure_date_with_transfers,
    currency
from $min_price_by_company
group by from_id, to_id, national_version, company_code, currency
order by from_id, to_id, national_version, min_price, min_price_with_transfers
limit 1000000
;
"""


class RunYqlException(Exception):
    pass


class MinPricesByAirlineCollector(object):
    def __init__(self, yql_client, company_repository):
        # type: (YqlClient, CompanyRepository) -> None
        self._yql_client = yql_client
        self._company_repository = company_repository

    def collect(self):
        logger.info('Start collecting min prices by airline')

        today = date.today()
        query = MIN_PRICE_BY_AIRLINE_YQL_QUERY_TEMPLATE.format(
            logs_from_date=today - timedelta(30),
            logs_to_date=today,
        )
        yql_query = self._yql_client.query(query, syntax_version=1)
        yql_query.run()
        yql_query.get_results(wait=True)

        if not yql_query.is_ok:
            if yql_query.errors:
                for error in yql_query.errors:
                    logger.error(' - %s', str(error))
            raise RunYqlException('Error while running YQL')

        logger.info('YQL query is done')

        yql_query.table.fetch_full_data()
        for row in yql_query.table.rows:
            (
                from_id,
                to_id,
                national_version,
                company_code,
                min_price,
                departure_date,
                min_price_with_transfers,
                departure_date_with_transfers,
                currency,
            ) = row
            company = self._company_repository.get_company_by_code(company_code)
            if company:
                yield (
                    int(from_id[1:]),
                    int(to_id[1:]),
                    national_version,
                    company.Id,
                    min_price,
                    departure_date,
                    min_price_with_transfers,
                    departure_date_with_transfers,
                    currency
                )
