# -*- coding: utf-8 -*-
import os
import time
from typing import Optional
from datetime import date, datetime

import ydb

from travel.avia.library.python.ydb.session_manager import YdbSessionManager


class MinPricesByAirlineTable(object):
    def __init__(self, ydb_session_manager, database, table_name):
        # type: (YdbSessionManager, str, str) -> None
        self._session_manager = ydb_session_manager
        self._database = database
        self._table_name = table_name

    def create_if_doesnt_exist(self):
        def callee(session):
            # type: (ydb.table.Session) -> None
            primary_key = [
                'from_id', 'to_id', 'national_version', 'airline_id',
            ]
            profile = (
                ydb.TableProfile()
                    .with_replication_policy(
                    ydb.ReplicationPolicy()
                        .with_allow_promotion(ydb.FeatureFlag.ENABLED)
                        .with_create_per_availability_zone(ydb.FeatureFlag.ENABLED)
                        .with_replicas_count(1)
                )
            )
            session.create_table(
                os.path.join(self._database, self._table_name),
                ydb.TableDescription()
                    .with_column(ydb.Column('from_id', ydb.OptionalType(ydb.PrimitiveType.Uint32)))
                    .with_column(ydb.Column('to_id', ydb.OptionalType(ydb.PrimitiveType.Uint32)))
                    .with_column(ydb.Column('national_version', ydb.OptionalType(ydb.PrimitiveType.Utf8)))
                    .with_column(ydb.Column('airline_id', ydb.OptionalType(ydb.PrimitiveType.Uint32)))
                    .with_column(ydb.Column('min_price', ydb.OptionalType(ydb.PrimitiveType.Uint32)))
                    .with_column(ydb.Column('departure_date', ydb.OptionalType(ydb.PrimitiveType.Utf8)))
                    .with_column(ydb.Column('min_price_with_transfers', ydb.OptionalType(ydb.PrimitiveType.Uint32)))
                    .with_column(ydb.Column('departure_date_with_transfers', ydb.OptionalType(ydb.PrimitiveType.Utf8)))
                    .with_column(ydb.Column('currency', ydb.OptionalType(ydb.PrimitiveType.Utf8)))
                    .with_column(ydb.Column('expires_at', ydb.OptionalType(ydb.DataType.Datetime)))
                    .with_primary_keys(*primary_key)
                    .with_index(ydb.TableIndex('expires_at_index').with_index_columns('expires_at'))
                    .with_profile(profile)
                    .with_ttl(ydb.TtlSettings().with_date_type_column('expires_at'))
            )

        with self._session_manager.get_session_pool() as session_pool:
            session_pool.retry_operation_sync(callee)

    def replace_batch(self, batch):
        query = """
        PRAGMA TablePathPrefix("{path}");

        DECLARE $data AS "List<Struct<
            from_id: Uint32,
            to_id: Uint32,
            national_version: Utf8,
            airline_id: Uint32,
            min_price: Uint32?,
            departure_date: Utf8?,
            min_price_with_transfers: Uint32?,
            departure_date_with_transfers: Utf8?,
            currency: Utf8,
            expires_at: Datetime>>";

        REPLACE INTO {table}
        SELECT * FROM AS_TABLE($data);
        """.format(path=self._database, table=self._table_name)

        def callee(session):
            # type: (ydb.table.Session) -> None
            prepared_query = session.prepare(query)
            session.transaction(ydb.SerializableReadWrite()).execute(
                prepared_query,
                commit_tx=True,
                parameters={
                    '$data': batch,
                },
                settings=ydb.table.settings.BaseRequestSettings().with_operation_timeout(5),
            )

        with self._session_manager.get_session_pool() as session_pool:
            session_pool.retry_operation_sync(callee)

    def delete_old(self, expired_date, timeout=120):
        # type: (date, int) -> None
        query = """
        PRAGMA TablePathPrefix("{path}");

        DECLARE $date as Utf8;

        DELETE FROM {table}
        WHERE departure_date <= $date or departure_date_with_transfers <= $date;
        """.format(path=self._database, table=self._table_name)

        def callee(session):
            # type: (ydb.table.Session) -> None
            prepared_query = session.prepare(query)
            session.transaction(ydb.SerializableReadWrite()).execute(
                prepared_query,
                commit_tx=True,
                parameters={
                    '$date': expired_date.strftime('%Y-%m-%d'),
                },
                settings=ydb.table.settings.BaseRequestSettings().with_operation_timeout(timeout),
            )

        with self._session_manager.get_session_pool() as session_pool:
            session_pool.retry_operation_sync(callee)


class MinPriceByAirline(object):
    __slots__ = (
        'from_id',
        'to_id',
        'national_version',
        'airline_id',
        'min_price',
        'departure_date',
        'min_price_with_transfers',
        'departure_date_with_transfers',
        'currency',
        'expires_at',
    )

    def __init__(
        self,
        from_id,
        to_id,
        national_version,
        airline_id,
        min_price,
        departure_date,
        min_price_with_transfers,
        departure_date_with_transfers,
        currency,
        expires_at,
    ):
        # type: (int, int, str, int, int, Optional[str], int, Optional[str], str, datetime) -> None
        self.from_id = from_id
        self.to_id = to_id
        self.national_version = national_version
        self.airline_id = airline_id
        self.min_price = min_price
        self.departure_date = departure_date
        self.min_price_with_transfers = min_price_with_transfers
        self.departure_date_with_transfers = departure_date_with_transfers
        self.currency = currency
        self.expires_at = int(time.mktime(expires_at.utctimetuple()))

    def __str__(self):
        return str({s: getattr(self, s) for s in self.__slots__})
