# -*- coding: utf-8 -*-
import logging
from datetime import datetime, date
from argparse import ArgumentParser

import ydb
from yql.api.v1.client import YqlClient

from travel.avia.library.python.lib_yt.client import configured_client
from travel.avia.library.python.references.station import create_station_cache
from travel.avia.library.python.references.station_to_settlement import create_station_to_settlement_cache
from travel.avia.avia_statistics.popular_months_by_route_updater.lib.updater import PopularMonthsWithPriceUpdater
from travel.avia.avia_statistics.popular_months_by_route_updater.lib.collector import PopularMonthsCollector
from travel.avia.avia_statistics.popular_months_by_route_updater.lib.table import PopularMonthWithPriceTable
from travel.avia.library.python.ydb.session_manager import YdbSessionManager


def main():
    parser = ArgumentParser()

    parser.add_argument('--debug', action='store_true', default=False)

    parser.add_argument('--yt-token', help='YT token', required=True)
    parser.add_argument('--yt-proxy', default='hahn')

    parser.add_argument('--yql-token', help='YQL token', required=True)

    parser.add_argument('--ydb-cluster', default='ydb-ru-prestable.yandex.net:2135')
    parser.add_argument('--ydb-database', default='/ru-prestable/ticket/testing/avia_statistics')
    parser.add_argument('--ydb-table', default='popular_months')
    parser.add_argument('--ydb-token', help='YDB token', required=True)

    parser.add_argument('--batch', help='Batch size', default=1000)

    parser.add_argument('--redirects-log-directory', help='Avia redirects log directory',
                        default='//home/avia/logs/avia-redir-balance-by-day-log')
    parser.add_argument('--date', help='Date', default=date.today().isoformat())

    arguments = parser.parse_args()

    logging.basicConfig(level=logging.INFO)

    if arguments.debug:
        logging.basicConfig(level=logging.DEBUG)

    yt_client = configured_client(proxy=arguments.yt_proxy, token=arguments.yt_token)
    yql_client = YqlClient(token=arguments.yql_token)
    station_cache = create_station_cache(yt_client)
    station_to_settlement_cache = create_station_to_settlement_cache(yt_client)
    collector = PopularMonthsCollector(
        yql_client,
        arguments.redirects_log_directory,
        station_cache,
        station_to_settlement_cache,
    )
    ydb_config = ydb.DriverConfig(
        endpoint=arguments.ydb_cluster,
        database=arguments.ydb_database,
        auth_token=arguments.ydb_token,
    )

    session_manager = YdbSessionManager(ydb_config)
    popular_months_table = PopularMonthWithPriceTable(session_manager, arguments.ydb_database, arguments.ydb_table)
    popular_months_updater = PopularMonthsWithPriceUpdater(
        collector,
        popular_months_table,
        arguments.batch,
    )
    target_date = datetime.strptime(arguments.date, '%Y-%m-%d').date()
    popular_months_updater.update_popular_months(target_date)


if __name__ == '__main__':
    main()
