# -*- coding: utf-8 -*-
import logging
from argparse import ArgumentParser

import ydb
from sandbox.projects.Travel.resources import dicts
from sandbox.projects.avia_resources.resources import AVIA_WIZARD_DIRECTION_NATIONAL
from sandbox.common.rest import Client as SandboxClient
from sandbox.common.proxy import OAuth

from travel.avia.library.python.lib_yt.client import configured_client
from travel.library.python.dicts.avia.direction_national_repository import DirectionNationalRepository
from travel.library.python.dicts.avia.settlement_repository import SettlementRepository
from travel.library.python.dicts.avia.station_repository import StationRepository
from travel.avia.avia_statistics.route_infos_updater.lib.updater import RouteInfosUpdater
from travel.avia.avia_statistics.route_infos_updater.lib.table import RouteInfosTable
from travel.avia.avia_statistics.lib.settlements_geo_index import SettlementsGeoIndex
from travel.avia.avia_statistics.landing_routes import get_landing_routes
from travel.avia.library.python.ydb.session_manager import YdbSessionManager
from travel.avia.library.python.sandbox.resource_fetcher import ResourceFetcher
from travel.avia.library.python.references.station_to_settlement import create_station_to_settlement_cache
from travel.avia.library.python.references.station import create_station_cache
from travel.avia.library.python.shared_flights_client.client import SharedFlightsClient

logger = logging.getLogger(__name__)


def main():
    parser = ArgumentParser()

    parser.add_argument('--debug', action='store_true', default=False)
    parser.add_argument(
        '--shared-flights-api-host',
        help='shared flights api base url',
        default='http://shared-flights.testing.avia.yandex.net',
    )

    parser.add_argument('--yt-token', help='YT token', required=True)
    parser.add_argument('--yt-proxy', default='hahn')

    parser.add_argument('--ydb-cluster', default='ydb-ru-prestable.yandex.net:2135')
    parser.add_argument('--ydb-database', default='/ru-prestable/ticket/testing/avia_statistics')
    parser.add_argument('--ydb-table', default='route_infos')
    parser.add_argument('--ydb-token', help='YDB token', required=True)

    parser.add_argument('--sandbox-token', help='Sandbox token', required=True)

    parser.add_argument('--batch', help='Batch size', default=100)

    arguments = parser.parse_args()

    logging.basicConfig(level=logging.INFO)

    if arguments.debug:
        logging.basicConfig(level=logging.DEBUG)

    sandbox_oauth = OAuth(arguments.sandbox_token)
    sandbox_client = SandboxClient(auth=sandbox_oauth)
    resource_fetcher = ResourceFetcher(sandbox_client, sandbox_oauth)

    settlement_repository = SettlementRepository()
    settlement_resource_data = resource_fetcher.fetch_latest_ready(dicts.TRAVEL_DICT_AVIA_SETTLEMENTS_PROD)
    settlement_repository.load_from_string(settlement_resource_data)

    station_repository = StationRepository()
    station_resource_data = resource_fetcher.fetch_latest_ready(dicts.TRAVEL_DICT_AVIA_STATION_PROD)
    station_repository.load_from_string(station_resource_data)

    direction_national_repository = DirectionNationalRepository()
    direction_national_resource_data = resource_fetcher.fetch_latest_ready(AVIA_WIZARD_DIRECTION_NATIONAL)
    direction_national_repository.load_from_string(direction_national_resource_data)

    settlements_geo_index = SettlementsGeoIndex(direction_national_repository, settlement_repository)

    yt_client = configured_client(proxy=arguments.yt_proxy, token=arguments.yt_token)
    ydb_config = ydb.DriverConfig(
        endpoint=arguments.ydb_cluster,
        database=arguments.ydb_database,
        auth_token=arguments.ydb_token,
    )
    session_manager = YdbSessionManager(ydb_config)

    route_infos_table = RouteInfosTable(session_manager, arguments.ydb_database, arguments.ydb_table)
    landing_routes = get_landing_routes(yt_client)
    station_to_settlement_cache = create_station_to_settlement_cache(yt_client)
    station_cache = create_station_cache(yt_client)
    shared_flights_client = SharedFlightsClient(arguments.shared_flights_api_host)
    route_infos_updater = RouteInfosUpdater(
        landing_routes,
        station_repository,
        station_to_settlement_cache,
        station_cache,
        settlements_geo_index,
        route_infos_table,
        shared_flights_client,
        arguments.batch,
    )
    route_infos_updater.update()


if __name__ == '__main__':
    main()
