# -*- coding: utf-8 -*-
import logging
from datetime import date, timedelta

from yql.api.v1.client import YqlClient

from travel.library.python.dicts.avia.company_repository import CompanyRepository


logger = logging.getLogger(__name__)

TOP_AIRLINES_BY_ROUTE_YQL_QUERY_TEMPLATE = """
use hahn;
pragma yson.DisableStrict;

$logs_from_date = '{logs_from_date}';
$logs_to_date = '{logs_to_date}';

$departure_date_from = CAST(CurrentUtcDateTime() as Uint32);
$departure_date_from = $departure_date_from - $departure_date_from % (24 * 60 * 60);
$departure_date_to = $departure_date_from + 6 * 31 * 24 * 60 * 60;


$get_company_code = ($segments) -> {{
    $flight_number = CAST(Yson::LookupString($segments[0], "route") as Utf8);
    return Unicode::Substring($flight_number, 0, 2);
}};

$single_company_segments = ($segments) -> {{
    return ListLength(ListUniq(ListMap($segments, ($s) -> {{
        $flight_number = CAST(Yson::LookupString($s, "route") as Utf8);
        return Unicode::Substring($flight_number, 0, 2);
    }}))) = 1;
}};

$segments_key = ($segments) -> {{
    return String::JoinFromList(ListMap($segments, ($s) -> {{ return CAST(Yson::LookupString($s, "route") as Utf8); }}), ".");
}};

$to_flight_numbers = ($segments) -> {{
    return ListMap($segments, ($s) -> {{
        return CAST(Yson::LookupString($s, "route") as Utf8);
    }});
}};

$fmt = DateTime::Format("%Y-%m-%d");

$variantsLog = (
    select
        Yson::ConvertToList(Yson::Parse(forward_segments)) as forward_segments,
        Yson::ConvertToList(Yson::Parse(backward_segments)) as backward_segments,
        from_settlement_id,
        to_settlement_id,
        forward_date,
        backward_date,
    from range('//home/logfeller/logs/avia-variants-log/1d', $logs_from_date, $logs_to_date)
);

$variants = (
    select
        $to_flight_numbers(forward_segments) as segments,
        from_settlement_id as from_id,
        to_settlement_id as to_id,
        $fmt(DateTime::FromSeconds(CAST(forward_date as Uint32))) as forward_date,
        $segments_key(forward_segments) as segments_key,
        $get_company_code(forward_segments) as company_code,
    from $variantsLog
    where
        from_settlement_id is not null and
        to_settlement_id is not null and
        $single_company_segments(forward_segments) and
        forward_date >= $departure_date_from and
        forward_date <= $departure_date_to
    UNION ALL
    select
        $to_flight_numbers(backward_segments) as segments,
        from_settlement_id as from_id,
        to_settlement_id as to_id,
        $fmt(DateTime::FromSeconds(CAST(backward_date as Uint32))) as forward_date,
        $segments_key(backward_segments) as segments_key,
        $get_company_code(backward_segments) as company_code,
    from $variantsLog
    where
        from_settlement_id is not null and
        to_settlement_id is not null and
        $single_company_segments(backward_segments) and
        backward_date is not null and
        forward_date >= $departure_date_from and
        forward_date <= $departure_date_to
);


$agg_factory = AggregationFactory("AGGREGATE_LIST_DISTINCT");
$agg_flatten = AggregateFlatten($agg_factory);

$all_single_company_options = (
    select
        from_id,
        to_id,
        forward_date,
        company_code,
        segments_key,
        AggregateBy(segments, $agg_flatten) as segments,
    from $variants
    group by from_id, to_id, forward_date, company_code, segments_key
);

$top_company_codes_by_route = (
    select
        from_id,
        to_id,
        company_code,
        ListLength(AGGREGATE_LIST_DISTINCT(segments_key)) as possible_options_count,
    from $all_single_company_options
    group by from_id, to_id, company_code
    order by possible_options_count desc
    limit 100000
);

select
    CAST(from_id as Uint32) as from_id,
    CAST(to_id as Uint32) as to_id,
    "ru" as national_version,
    AGGREGATE_LIST(company_code) as popular_companies
from $top_company_codes_by_route
group by from_id, to_id
order by from_id, to_id
;
"""


class RunYqlException(Exception):
    pass


class TopAirlinesCollector(object):
    def __init__(self, yql_client, company_repository):
        # type: (YqlClient, CompanyRepository) -> None
        self._yql_client = yql_client
        self._company_repository = company_repository

    def collect(self):
        logger.info('Start collecting top airlines')

        today = date.today()
        query = TOP_AIRLINES_BY_ROUTE_YQL_QUERY_TEMPLATE.format(
            logs_from_date=today - timedelta(45),
            logs_to_date=today,
        )
        yql_query = self._yql_client.query(query, syntax_version=1)
        yql_query.run()
        yql_query.get_results(wait=True)

        if not yql_query.is_ok:
            if yql_query.errors:
                for error in yql_query.errors:
                    logger.error(' - %s', str(error))
            raise RunYqlException('Error while running YQL')

        logger.info('YQL query is done')

        yql_query.table.fetch_full_data()
        for from_id, to_id, national_version, popular_companies in yql_query.table.rows:
            yield (
                from_id,
                to_id,
                national_version,
                [
                    c.Id
                    for c in filter(None, [self._company_repository.get_company_by_code(c) for c in popular_companies])
                ],
            )
