# -*- coding: utf-8 -*-
import logging
from datetime import datetime, date
from argparse import ArgumentParser

import ydb
from sandbox.projects.Travel.resources import dicts
from sandbox.common.rest import Client as SandboxClient
from sandbox.common.proxy import OAuth

from travel.avia.library.python.backend_client import BackendClient
from travel.avia.library.python.lib_yt.client import configured_client
from travel.library.python.dicts.avia.settlement_repository import SettlementRepository
from travel.avia.avia_statistics.updaters.city_to.city_route_crosslinks_updater.lib.updater import (
    CityRouteCrosslinksUpdater
)
from travel.avia.avia_statistics.updaters.city_to.city_route_crosslinks_updater.lib.table import (
    CityRouteCrosslinksTable
)
from travel.avia.avia_statistics.landing_cities import iter_landing_cities
from travel.avia.avia_statistics.landing_routes import get_route_weights, iter_landing_routes
from travel.avia.avia_statistics.services.price_index.client import Client as PriceIndexClient
from travel.avia.library.python.ydb.session_manager import YdbSessionManager
from travel.avia.library.python.sandbox.resource_fetcher import ResourceFetcher

logger = logging.getLogger(__name__)


def main():
    parser = ArgumentParser()

    parser.add_argument('--debug', action='store_true', default=False)
    parser.add_argument(
        '--avia-backend-host',
        help='avia-backend host',
        default='http://backend.production.avia.yandex.net',
    )
    parser.add_argument(
        '--price-index-host',
        help='price-index host',
        default='http://price-index.production.avia.yandex.net',
    )
    parser.add_argument('--yt-token', help='YT token', required=True)
    parser.add_argument('--yt-proxy', default='hahn')

    parser.add_argument('--ydb-cluster', default='ydb-ru-prestable.yandex.net:2135')
    parser.add_argument('--ydb-database', default='/ru-prestable/ticket/testing/avia_statistics')
    parser.add_argument('--ydb-table', default='city_to_route_crosslinks')
    parser.add_argument('--ydb-token', help='YDB token', required=True)

    parser.add_argument('--sandbox-token', help='Sandbox token', required=True)

    parser.add_argument('--price-index-batch', help='price-index request batch size', default=100)
    parser.add_argument('--ydb-batch', help='YDB upsert request batch size', default=1000)
    parser.add_argument('--window-size', help='Price searching window size in the future', default=30)
    parser.add_argument('--date', help='Date', default=date.today().isoformat())
    parser.add_argument(
        '--save-crosslinks',
        help='Enables saving JSON file with crosslinks',
        action='store_true',
        default=False,
    )

    arguments = parser.parse_args()

    logging.basicConfig(level=logging.INFO)

    if arguments.debug:
        logging.basicConfig(level=logging.DEBUG)

    sandbox_oauth = OAuth(arguments.sandbox_token)
    sandbox_client = SandboxClient(auth=sandbox_oauth)
    resource_fetcher = ResourceFetcher(sandbox_client, sandbox_oauth)

    settlement_repository = SettlementRepository()
    settlement_resource_data = resource_fetcher.fetch_latest_ready(dicts.TRAVEL_DICT_AVIA_SETTLEMENTS_PROD)
    settlement_repository.load_from_string(settlement_resource_data)

    yt_client = configured_client(proxy=arguments.yt_proxy, token=arguments.yt_token)
    ydb_config = ydb.DriverConfig(
        endpoint=arguments.ydb_cluster,
        database=arguments.ydb_database,
        auth_token=arguments.ydb_token,
    )
    session_manager = YdbSessionManager(ydb_config)

    city_route_crosslinks_table = CityRouteCrosslinksTable(session_manager, arguments.ydb_database, arguments.ydb_table)
    price_index_client = PriceIndexClient(arguments.price_index_host)
    backend_client = BackendClient(arguments.avia_backend_host)
    landing_cities = list(iter_landing_cities(yt_client))
    landing_routes = list(iter_landing_routes(yt_client))[::-1]
    route_weights = get_route_weights(yt_client)
    city_route_crosslinks_updater = CityRouteCrosslinksUpdater(
        yt_client,
        landing_cities,
        landing_routes,
        route_weights,
        settlement_repository,
        city_route_crosslinks_table,
        backend_client,
        price_index_client,
        ydb_batch_size=arguments.ydb_batch,
    )
    target_date = datetime.strptime(arguments.date, '%Y-%m-%d')
    city_route_crosslinks_updater.update(
        target_date,
        arguments.window_size,
        arguments.save_crosslinks,
    )


if __name__ == '__main__':
    main()
