# -*- coding: utf-8 -*-
import logging
from collections import defaultdict

import numpy as np

from travel.avia.avia_statistics.landing_cities import LandingRoute, LandingCity
from travel.avia.avia_statistics.landing_routes import RouteWeight  # noqa
from travel.avia.avia_statistics.lib.consts import RUSSIA_COUNTRY_ID

logger = logging.getLogger(__name__)


class CrosslinksProvider(object):
    CROSSLINKS_PER_PAGE = 10

    def __init__(
        self,
        landing_cities,
        landing_routes,
        route_weights,
        settlement_repository,
        avia_backend_client,
        yt_client
    ):
        """
        :param typing.List[LandingCity] landing_cities:
        :param typing.List[LandingRoute] landing_routes:
        :param typing.Dict[LandingRoute, RouteWeight] route_weights:
        :param travel.library.python.dicts.avia.settlement_repository.SettlementRepository settlement_repository:
        :param travel.avia.library.python.backend_client.client.Client avia_backend_client:
        :param yt.wrapper.YtClient yt_client:
        """

        self._landing_cities = landing_cities
        self._landing_routes = landing_routes
        self._route_weights = route_weights
        self._settlement_repository = settlement_repository
        self._avia_backend_client = avia_backend_client
        self._yt_client = yt_client

    def generate(self):
        crosslinks = self._generate_crosslinks()
        logger.debug('crosslinks count = %d', len(crosslinks))
        logger.debug('non empty crosslinks count = %d', sum(len(v) != 0 for v in crosslinks.values()))
        logger.debug('empty crosslinks count = %d', sum(len(v) == 0 for v in crosslinks.values()))
        logger.debug('fullfilled pages = %d', sum(len(v) >= self.CROSSLINKS_PER_PAGE for v in crosslinks.values()))
        return crosslinks

    def _generate_crosslinks(self):
        landing_routes_by_national_version = defaultdict(set)
        for route in self._landing_routes:
            landing_routes_by_national_version[route.national_version].add(route)

        crosslinks_by_route = {}
        for nv, landing_routes in landing_routes_by_national_version.items():
            logger.info('Start work on %s', nv)
            logger.info('\t%s routes', len(landing_routes))
            popular_directions = self._get_popular_directions(nv)
            logger.info('\t%s cities with popular directions', len(popular_directions))
            crosslinks = self._generate_route_crosslinks_for_cities_with_same_national_version(
                self._landing_cities, landing_routes, popular_directions
            )
            logger.info('\t%s cities with crosslinks', len(popular_directions))
            crosslinks_by_route.update(crosslinks)
        return crosslinks_by_route

    def _get_popular_directions(self, national_version):
        """
        :param str national_version:
        :rtype: Dict[LandingCity, typing.List[LandingRoute]]
        :return: Направления для города и нац. версии, упорядоченые по популярности.
        """
        popular_to_city_routes = defaultdict(list)
        directions = self._avia_backend_client.direction(national_version=national_version)
        for direction in directions:
            city = LandingCity(direction['arrival_settlement_id'], national_version)
            route = LandingRoute(
                direction['departure_settlement_id'], direction['arrival_settlement_id'], national_version
            )
            popular_to_city_routes[city].append(route)
        return dict(popular_to_city_routes)

    def _generate_route_crosslinks_for_cities_with_same_national_version(
        self, landing_cities, landing_routes, popular_directions
    ):
        """
        Добавляем популярные направления для города прилета, на которые есть ссылка в routes
        Добавляем случайными направлениями в город, если не набрали 10 с учетом числа переходов
        Первыми берем города вылета из России, тут не только для зарубежных, но и для отечественных
        :type landing_routes: typing.Set[LandingRoute]
        """
        logger.info('Start generating crosslinks')
        crosslinks = defaultdict(list)

        for city in landing_cities:
            city_popular_directions = popular_directions.get(city, [])
            city_popular_directions = filter(landing_routes.__contains__, city_popular_directions)

            crosslinks[city].extend(city_popular_directions)

        self._fill_with_random_routes(landing_cities, landing_routes, crosslinks)

        self._prioritize_departures_from_Russia(crosslinks)

        return dict(crosslinks)

    def _fill_with_random_routes(self, landing_cities, landing_routes, crosslinks):
        logger.info('Fill uncompleted cities with random routes')
        cities_to_be_filled = {
            city for city in landing_cities if not self._check_completeness(crosslinks, city)
        }
        logger.info('Total cities to be filled: %d', len(cities_to_be_filled))

        possible_city_routes = defaultdict(list)
        for route in landing_routes:
            city = LandingCity(route.to_id, route.national_version)
            if city not in cities_to_be_filled or route in crosslinks[city]:
                continue

            possible_city_routes[city].append(route)

        for i, city in enumerate(cities_to_be_filled):
            if i and i % 100 == 0:
                logger.info('Processed: %d', i)

            possible_routes = possible_city_routes[city]
            if len(possible_routes) + len(crosslinks[city]) <= self.CROSSLINKS_PER_PAGE:
                crosslinks[city].extend(possible_routes)
                continue

            total_redirs = sum(self._route_weights[r].redirs for r in possible_routes)
            proba_by_route = {r: float(self._route_weights[r].redirs) / total_redirs for r in possible_routes}
            probas_distribution = [proba_by_route[r] for r in possible_routes]
            to_ad_cnt = self.CROSSLINKS_PER_PAGE - len(crosslinks[city])

            random_routes_indices = np.random.choice(
                range(len(possible_routes)), to_ad_cnt, replace=False, p=probas_distribution
            )
            random_routes = map(possible_routes.__getitem__, random_routes_indices)
            crosslinks[city].extend(list(random_routes))

    def _prioritize_departures_from_Russia(self, crosslinks):
        for city, routes in crosslinks.items():
            routes.sort(
                key=(lambda route: self._settlement_repository.get(route.from_id).CountryId != RUSSIA_COUNTRY_ID)
            )

    def _check_completeness(self, crosslinks, city):
        if len(crosslinks[city]) >= self.CROSSLINKS_PER_PAGE:
            crosslinks[city] = crosslinks[city][:self.CROSSLINKS_PER_PAGE]
            return True
        return False


class RoutesDict(object):
    @classmethod
    def as_json(cls, routes_dict):
        return {cls.map_key(k): map(cls.map_value, v) for k, v in routes_dict.items()}

    @staticmethod
    def map_key(k):
        # type: (LandingCity) -> str
        return '{}_{}'.format(k.to_id, k.national_version)

    @staticmethod
    def map_value(value):
        # type: (LandingRoute) -> typing.Dict
        return {
            'from_id': value.from_id,
            'to_id': value.to_id,
            'national_version': value.national_version,
        }
