# -*- coding: utf-8 -*-
from datetime import date
from dateutil.relativedelta import relativedelta
import logging

from yql.api.v1.client import YqlClient  # noqa

from travel.avia.avia_statistics.lib.consts import CURRENCY_BY_NATIONAL_VERSION

logger = logging.getLogger(__name__)

MONTH_AND_YEAR_PRICES_BY_CITY_TO_QUERY_TEMPLATE = """
use hahn;
pragma yson.DisableStrict;
pragma yt.InferSchema;

$logs_date_from = '{logs_date_from}';
$logs_date_to = '{logs_date_to}';

$point_key_to_id = ($pk) -> {{
    return CAST(SUBSTRING($pk, 1) as Uint32)
}};

$parse = DateTime::Parse("%Y-%m-%d");

$fmt = DateTime::Format("%Y-%m-%d");

$current_month_first_day_year_ago = $fmt(DateTime::MakeDate(DateTime::StartOfMonth(DateTime::MakeDate(DateTime::Update(
    CurrentUtcDate(),
    DateTime::GetYear(CurrentUtcDate()) - 1us AS Year
)))));

$previous_month_last_day = $fmt(DateTime::MakeDate(DateTime::StartOfMonth(CurrentUtcDate())) - DateTime::IntervalFromDays(1));

$currency_by_nv = AsDict({currency_by_nv});

$redirs = (
    select
        $point_key_to_id(FROMID) as from_id,
        $point_key_to_id(TOID) as to_id,
        NATIONAL_VERSION as national_version,
        `WHEN` as forward_date,
        ORIGINAL_PRICE as price,
        ORIGINAL_CURRENCY as currency,
        DateTime::GetYear($parse(`WHEN`)) as year,
        DateTime::GetMonth($parse(`WHEN`)) as month
    from range(`{redirects_log_path}`, $logs_date_from, $logs_date_to)
    WHERE
        `FILTER` = 0
        and KLASS = 'economy'
        and (`WHEN` between $current_month_first_day_year_ago and $previous_month_last_day)
        and DictContains($currency_by_nv, NATIONAL_VERSION)
        and $currency_by_nv[NATIONAL_VERSION] = ORIGINAL_CURRENCY
);

$year_median_by_city_to = (
    select
        to_id,
        national_version,
        CAST(median(price) as Uint32) as year_median_price,
    from $redirs
    group by to_id, national_version
);

$month_medians_by_city_to = (
    select
        to_id,
        national_version,
        year,
        month,
        CAST(median(price) as Uint32) as month_median_price,
        COUNT(*) as redirs_count
    from $redirs
    group by to_id, national_version, year, month
);

$popular_month = (
    select
        to_id,
        national_version,
        -MAX(AsTuple(redirs_count, -month_median_price, month, year)).1 as popular_month_median_price,
        MAX(AsTuple(redirs_count, -month_median_price, month, year)).2 as popular_month,
        MAX(AsTuple(redirs_count, -month_median_price, month, year)).3 as popular_month_year,
    from $month_medians_by_city_to
    group by to_id, national_version
);

$min_by_month = (
    select
        to_id,
        national_version,
        MIN(AsTuple(month_median_price, year, month)).0 as min_month_median_price,
        MIN(AsTuple(month_median_price, year, month)).1 as min_year,
        MIN(AsTuple(month_median_price, year, month)).2 as min_month,
    from $month_medians_by_city_to

    group by to_id, national_version
);

$max_by_month = (
    select
        to_id,
        national_version,
        MAX(AsTuple(month_median_price, year, month)).0 as max_month_median_price,
        MAX(AsTuple(month_median_price, year, month)).1 as max_year,
        MAX(AsTuple(month_median_price, year, month)).2 as max_month,
    from $month_medians_by_city_to

    group by to_id, national_version
);

select
    y.to_id as to_id,
    y.national_version as national_version,
    y.year_median_price as year_median_price,

    p.popular_month_year as popular_month_year,
    p.popular_month as popular_month,
    p.popular_month_median_price as popular_month_median_price,

    min_.min_year as min_month_year,
    min_.min_month as min_month,
    min_.min_month_median_price as min_month_median_price,

    max_.max_year as max_month_year,
    max_.max_month as max_month,
    max_.max_month_median_price as max_month_median_price,

from $year_median_by_city_to as y
join $popular_month as p
on y.to_id = p.to_id and y.national_version = p.national_version

join $min_by_month as min_
on y.to_id = min_.to_id and y.national_version = min_.national_version

join $max_by_month as max_
on y.to_id = max_.to_id and y.national_version = max_.national_version

join hahn.`{landing_cities_table}` as c
on y.to_id = c.id
WHERE y.national_version = 'ru'
"""


class RunYqlException(Exception):
    pass


class MonthAndYearPricesByCityToCollector(object):
    def __init__(
        self,
        yql_client,
        redirects_log_path,
        landing_cities_table,
    ):
        """
        :param YqlClient yql_client:
        :param str redirects_log_path:
        :param str landing_cities_table:
        """
        self._yql_client = yql_client
        self._redirects_log_path = redirects_log_path
        self._landing_cities_table = landing_cities_table

    def collect(self, today):
        # type: (date) -> str
        date_from = (today - relativedelta(years=1, months=6, day=1)).strftime('%Y-%m-%d')  # 1.5 year ago
        date_to = (today - relativedelta(days=today.day)).strftime('%Y-%m-%d')  # previous month last day
        logger.info(
            'calculating popular months for range (%s) - (%s) in %s',
            date_from,
            date_to,
            self._redirects_log_path,
        )
        query = MONTH_AND_YEAR_PRICES_BY_CITY_TO_QUERY_TEMPLATE.format(
            logs_date_from=date_from,
            logs_date_to=date_to,
            redirects_log_path=self._redirects_log_path,
            landing_cities_table=self._landing_cities_table,
            currency_by_nv=', '.join(
                'AsTuple(\'{}\', \'{}\')'.format(k, v)
                for k, v in CURRENCY_BY_NATIONAL_VERSION.items()
            ),
        )

        yql_query = self._yql_client.query(query, syntax_version=1)
        yql_query.run()
        yql_query.get_results(wait=True)

        if not yql_query.is_ok:
            if yql_query.errors:
                for error in yql_query.errors:
                    logger.error(' - %s', str(error))
            raise RunYqlException('Error while running YQL')

        logger.info('YQL query is done')

        yql_query.table.fetch_full_data()
        for row in yql_query.table.rows:
            yield row
