# -*- coding: utf-8 -*-
import logging
from datetime import date

from travel.avia.avia_statistics.lib.consts import CURRENCY_BY_NATIONAL_VERSION
from travel.avia.avia_statistics.updaters.city_to.month_and_year_prices_by_city_to_updater.lib.table import (
    MonthAndYearPricesByCityTo, MonthAndYearPricesByCityToTable
)
from travel.avia.avia_statistics.updaters.city_to.month_and_year_prices_by_city_to_updater.lib.collector import (
    MonthAndYearPricesByCityToCollector
)

logger = logging.getLogger(__name__)


class MonthAndYearPricesByCityToUpdater(object):
    def __init__(
        self,
        collector,
        table,
        batch_size=1000,
    ):
        # type: (MonthAndYearPricesByCityToCollector, MonthAndYearPricesByCityToTable, int) -> None
        self._collector = collector
        self._table = table
        self._batch_size = batch_size

    def update(self, today):
        # type: (date) -> None
        self._table.create_if_doesnt_exist()
        batch = []
        total_processed = 0
        logger.info('start collecting month and year prices by city to')
        for route_pop_month_price in self._collector.collect(today):
            batch.append(route_pop_month_price)
            if len(batch) == self._batch_size:
                self._process_batch(batch)
                total_processed += len(batch)
                logger.info('processed: %s', total_processed)
                batch = []
        if batch:
            self._process_batch(batch)
            total_processed += len(batch)
            logger.info('processed: %s', total_processed)
        logger.info('all %s month and year prices by city to were stored into YDB', total_processed)

    def _process_batch(self, batch):
        def map_to_record(route_with_popular_month_price):
            (
                to_id,
                national_version,
                year_median_price,

                popular_month_year,
                popular_month,
                popular_month_median_price,

                min_month_year,
                min_month,
                min_month_median_price,

                max_month_year,
                max_month,
                max_month_median_price,
            ) = route_with_popular_month_price

            currency = CURRENCY_BY_NATIONAL_VERSION.get(national_version)

            return MonthAndYearPricesByCityTo(
                to_id=to_id,
                national_version=national_version,
                currency=currency,
                year_median_price=year_median_price,
                popular_month_year=popular_month_year,
                popular_month=popular_month,
                popular_month_median_price=popular_month_median_price,
                min_month_year=min_month_year,
                min_month=min_month,
                min_month_median_price=min_month_median_price,
                max_month_year=max_month_year,
                max_month=max_month,
                max_month_median_price=max_month_median_price,
            )

        month_with_price_records = list(filter(None, map(map_to_record, batch)))
        self._table.replace_batch(month_with_price_records)
