import os

from travel.avia.library.python.ydb.session_manager import YdbSessionManager
import ydb


class DirectionPopularity:
    __slots__ = (
        'settlement_from_id',
        'settlement_to_id',
        'redir_number',
    )

    def __init__(self, settlement_from_id: int, settlement_to_id: int, redir_number: int):
        self.settlement_from_id = settlement_from_id
        self.settlement_to_id = settlement_to_id
        self.redir_number = redir_number

    def __str__(self):
        return str({s: getattr(self, s) for s in self.__slots__})


class DirectionPopularityTable:
    def __init__(self, session_manager: YdbSessionManager, database: str, table_name: str):
        self.database = database
        self.table_name = table_name
        self._session_manager = session_manager

    def create_if_doesnt_exist(self) -> None:
        def callee(session: ydb.table.Session) -> None:
            profile = ydb.TableProfile().with_replication_policy(
                ydb.ReplicationPolicy()
                .with_allow_promotion(ydb.FeatureFlag.ENABLED)
                .with_create_per_availability_zone(ydb.FeatureFlag.ENABLED)
                .with_replicas_count(1)
            )
            session.create_table(
                os.path.join(self.database, self.table_name),
                ydb.TableDescription()
                .with_column(ydb.Column('settlement_from_id', ydb.OptionalType(ydb.PrimitiveType.Uint32)))
                .with_column(ydb.Column('settlement_to_id', ydb.OptionalType(ydb.PrimitiveType.Uint32)))
                .with_column(ydb.Column('redir_number', ydb.OptionalType(ydb.PrimitiveType.Uint32)))
                .with_primary_keys('settlement_from_id', 'settlement_to_id')
                .with_profile(profile),
            )

        with self._session_manager.get_session_pool() as session_pool:
            session_pool.retry_operation_sync(callee)

    def replace_batch(self, batch: list[DirectionPopularity]) -> None:
        query = """
        --!syntax_v1
        PRAGMA TablePathPrefix("{path}");

        DECLARE $data AS List<Struct<
            settlement_from_id: Uint32,
            settlement_to_id: Uint32,
            redir_number: Uint32>>;

        REPLACE INTO {table}
        SELECT * FROM AS_TABLE($data);
        """.format(
            path=self.database, table=self.table_name
        )

        def callee(session: ydb.table.Session) -> None:
            prepared_query = session.prepare(query)
            session.transaction(ydb.SerializableReadWrite()).execute(
                prepared_query,
                commit_tx=True,
                parameters={
                    '$data': batch,
                },
                settings=ydb.table.settings.BaseRequestSettings().with_operation_timeout(5),
            )

        with self._session_manager.get_session_pool() as session_pool:
            session_pool.retry_operation_sync(callee)

    def count(self) -> int:
        query = """
        --!syntax_v1
        PRAGMA TablePathPrefix("{path}");

        SELECT COUNT(*) as c FROM {table};
        """.format(
            path=self.database, table=self.table_name
        )

        def callee(session: ydb.table.Session) -> int:
            result_sets = session.transaction(ydb.SerializableReadWrite()).execute(
                query,
                commit_tx=True,
                settings=ydb.table.settings.BaseRequestSettings().with_operation_timeout(5),
            )
            return result_sets[0].rows[0]['c']

        with self._session_manager.get_session_pool() as session_pool:
            return session_pool.retry_operation_sync(callee)
