# coding=utf-8
from __future__ import absolute_import

import gc
import os
import signal
import socket
import sys
import time

from travel.library.python.gunicorn.worker_watcher import WorkersWatcher
from travel.library.python.solomon.server import SolomonMetricsServer

from travel.avia.library.python.redis.ping import RedisBackgroundPing
from travel.avia.library.python.shared_objects import SharedFlag

from travel.avia.backend.main.lib.caches import default_cache, shared_cache
from travel.avia.backend.main.settings import (
    CACHEROOT,
    CACHE_REDIS,
    REDIS_MAXIMUM_LAG,
    REDIS_PING_SLEEP_TIME,
    SHARED_CACHE_REDIS,
)
from travel.avia.backend.main.settings import env_variable_provider, get_metric_queue


name = 'yandex-avia-backend'
bind = env_variable_provider.get('bind_address')
workers = int(env_variable_provider.get('gunicorn_workers_count'))

max_requests = int(os.getenv('GUNICORN_MAX_REQUESTS', 100))
max_requests_jitter = max_requests // 5

timeout = os.getenv('GUNICORN_TIMEOUT', 60)

sys.path[:0] = [os.path.abspath(os.path.join(os.path.dirname(__file__), '..', 'common', 'contrib'))]

os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'local_settings')

preload_app = True

worker_class = 'travel.library.python.gunicorn.worker_watcher.workers.SyncWorker'

RECACHE_FUNC_UID = 'travel.avia.library.python.common.precache.gunicorn.conf.recache'

VERSION = os.getenv('DEPLOY_DOCKER_IMAGE', ':unknown').rsplit(':', 1)[-1]
TRACING_SERVICE_NAME = 'avia-backend'
TRACING_JAEGER_SAMPLER_TIPE = os.getenv('JAEGER_SAMPLER_TYPE', 'probabilistic')
TRACING_JAEGER_SAMPLER_PARAMETER = float(os.getenv('JAEGER_SAMPLER_PARAMETER', 0.001))

shutdown_flag = SharedFlag()
cache_readiness_flag = SharedFlag()
shared_cache_readiness_flag = SharedFlag()

solomon_server = SolomonMetricsServer()
solomon_server.register_metric_queue(get_metric_queue())

key_ttl = 60
cache_ping = RedisBackgroundPing(
    default_cache,
    '{}ping_check/{}'.format(CACHEROOT, socket.gethostname()),
    key_ttl,
    REDIS_PING_SLEEP_TIME,
    REDIS_MAXIMUM_LAG,
    cache_readiness_flag,
)
if SHARED_CACHE_REDIS:
    shared_cache_ping = RedisBackgroundPing(
        shared_cache,
        '{}shared_ping_check/{}'.format(CACHEROOT, socket.gethostname()),
        key_ttl,
        REDIS_PING_SLEEP_TIME,
        0,
        shared_cache_readiness_flag,
    )

watcher = WorkersWatcher()


def precache(server):
    """
    # Send signal to recache:
    from travel.avia.library.python.common.utils.mysql_switcher import signals as switcher_signals
    switcher_signals.data_updated.send(cls)
    """

    from travel.avia.library.python.common.precache.backend import _precache
    from travel.avia.library.python.common.utils.mysql_switcher.signals import data_updated
    from travel.avia.backend.repository import (
        airline_repository,
        national_version_repository,
        partner_repository,
        partner_popularity_repository,
        route_partner_popularity_repository,
        airline_rating_repository,
        similar_airlines_repository,
        currency_repository,
        currency_rates_repository,
        currency_translation_repository,
        recipes_repository,
        holiday_repository,
        settlement_repository,
        settlement_geo_index,
        country_repository,
        region_repository,
        station_repository,
        geo_relations_repository,
        plane_transport_model_repository,
        station_type_repository,
        review_stat_repository,
        flights_to_city_airlines_repository,
        direction_repository,
        flight_rating_repository,
        iata_correction_repository,
        company_repository,
        settlement_big_image_repository,
        covid_info_repository,
    )
    from travel.avia.backend.main.lib.display_code_manager import display_code_manager

    server_pid = os.getpid()

    def recache(**kwargs):
        server.log.info('Data update detected, recaching')

        os.kill(server_pid, signal.SIGHUP)

    while True:
        try:
            server.log.info('Precaching...')

            data_updated.disconnect(dispatch_uid=RECACHE_FUNC_UID)

            _precache()
            # pre cache repositories
            national_version_repository.pre_cache()
            partner_repository.pre_cache()
            partner_popularity_repository.pre_cache()
            route_partner_popularity_repository.pre_cache()
            airline_repository.pre_cache()
            airline_rating_repository.pre_cache()
            similar_airlines_repository.pre_cache()
            currency_repository.fetch()
            currency_translation_repository.fetch()
            currency_rates_repository.fetch()
            recipes_repository.pre_cache()
            holiday_repository.pre_cache()
            settlement_repository.pre_cache()
            settlement_geo_index.pre_cache()
            country_repository.pre_cache()
            region_repository.pre_cache()
            station_repository.pre_cache()
            geo_relations_repository.pre_cache()
            display_code_manager.pre_cache()
            plane_transport_model_repository.pre_cache()
            station_type_repository.pre_cache()
            review_stat_repository.pre_cache()
            flights_to_city_airlines_repository.pre_cache()
            direction_repository.pre_cache()
            flight_rating_repository.pre_cache()
            iata_correction_repository.pre_cache()
            company_repository.pre_cache()
            settlement_big_image_repository.pre_cache()
            covid_info_repository.pre_cache()

            data_updated.connect(recache, weak=False, dispatch_uid=RECACHE_FUNC_UID)
        except Exception:
            server.log.error('Unhandled exception in precache (Retrying in 30 seconds):', exc_info=True)

            time.sleep(30)

            continue

        break

    server.log.info('Done')


def on_starting(server):
    server.log.info('Start master process ...')

    watcher.on_starting(server)

    solomon_server.register_gauge({'sensor': 'gunicorn.requests_in_work'}, watcher.get_requests_in_work_count)
    solomon_server.register_gauge({'sensor': 'gunicorn.requests_in_work_max'}, watcher.get_max_requests_in_work)
    solomon_server.run_in_thread()

    if CACHE_REDIS:
        cache_ping.run()
    if SHARED_CACHE_REDIS:
        shared_cache_ping.run()
    precache(server)
    server.log.info('... finish start master process')


def on_reload(server):
    server.log.info('Start recache...')
    if CACHE_REDIS:
        cache_ping.run()
    if SHARED_CACHE_REDIS:
        shared_cache_ping.run()
    precache(server)
    server.log.info('... finish recache')


def pre_fork(server, worker):
    # Закрываем соединение, чтобы не передавать воркерам копию сокета [RASPWIZARDS-175]
    from django.core.cache import cache
    cache.close()

    watcher.pre_fork(server, worker)

    # собираем мусор, чтобы не оставлять его в форках
    gc.collect()


def child_exit(server, worker):
    watcher.child_exit(server, worker)


def on_exit(server):
    if CACHE_REDIS:
        cache_ping.stop()
    if SHARED_CACHE_REDIS:
        shared_cache_ping.stop()
    solomon_server.stop_thread()


def post_fork(server, worker):
    from flask_opentracing_helpers import setup_tracing

    from travel.avia.backend.wsgi import application

    application.shutdown_flag = shutdown_flag
    application.cache_readiness_flag = cache_readiness_flag
    application.shared_cache_readiness_flag = shared_cache_readiness_flag
    if not SHARED_CACHE_REDIS:
        application.shared_cache_readiness_flag.set()

    server.log.info('Setup tracing...')
    try:
        setup_tracing(
            flask_application=application,
            service_name=TRACING_SERVICE_NAME,
            version=VERSION,
            include_env_variables=['DEPLOY_STAGE_ID', 'DEPLOY_NODE_DC', 'DEPLOY_BOX_ID'],
            filter_paths=['ping'],
            jaeger_sampler_type=TRACING_JAEGER_SAMPLER_TIPE,
            jaeger_sampler_parameter=TRACING_JAEGER_SAMPLER_PARAMETER,
        )
    except Exception:
        server.log.exception('Failed to setup tracing')
    else:
        server.log.info('Done setup tracing successfully')
