# -*- encoding: utf-8 -*-
from __future__ import absolute_import

from marshmallow import fields, ValidationError
from marshmallow.utils import get_value

from django.db import models

from travel.avia.library.python.common.models.geo import Point
from travel.avia.library.python.common.utils.date import timedelta2minutes

from travel.avia.backend.main.api.processors import process_fields
from travel.avia.backend.main.lib.exceptions import ApiException


def country_key_validator(value):
    if not value[0] in 'l':
        raise ValidationError('wrong country key')


def settlement_key_validator(value):
    if not value[0] in 'c':
        raise ValidationError('wrong settlement key')


def station_key_validator(value):
    if not value[0] in 's':
        raise ValidationError('wrong station key')


class Related(fields.Field):
    _CHECK_ATTRIBUTE = False

    def __init__(self, *args, **kwargs):
        super(Related, self).__init__(*args, **kwargs)

        self._handler = self.metadata.get('handler')

        if not self._handler:
            raise ValueError('Related field should have handler')

        self._params = self.metadata.get('params', {})
        self._extra_params = self.metadata.get('extra_params', {})

    def _serialize(self, value, attr, obj):
        if not obj:
            return None

        if self.context is None:
            msg = 'No context available for Related field "%s"' % attr
            return ApiException(msg).dump()

        fieldData = self.context['fields'].get(self.name)
        if fieldData is None:
            return None

        params = fieldData['params']
        fields = fieldData['fields']

        for to_key, from_key in self._params.items():
            if from_key == 'self':
                params[to_key] = obj
                continue

            if callable(from_key):
                params[to_key] = from_key(obj, self.context)
                continue

            params[to_key] = get_value(from_key, obj, None)

            # если понадобится - нужно будет отрефакторить добавив в параметры allow_none
            # пока таких кейсов нет и не вижу где бы понадобилось - не делаю
            if params[to_key] is None:
                return None

        params.update(self._extra_params)

        # при создании хендлера нужно ловить ошибки
        try:
            handler = self._handler(obj, params)
        except Exception, e:
            return ApiException(e).dump()

        # а вот внутри он уже сам поймает
        return handler(params, fields)


class PointKey(fields.String):
    def _deserialize(self, value, attr, data):
        try:
            return Point.parse_key(value)
        except ValueError:
            raise ValidationError('wrong point key %s' % value)


class CountryKey(fields.String):
    def __init__(self, *args, **kwargs):
        super(CountryKey, self).__init__(*args, **kwargs)

        self.validators.insert(0, country_key_validator)


class SettlementKey(fields.String):
    def __init__(self, *args, **kwargs):
        super(SettlementKey, self).__init__(*args, **kwargs)

        self.validators.insert(0, settlement_key_validator)


class StationKey(fields.String):
    def __init__(self, *args, **kwargs):
        super(StationKey, self).__init__(*args, **kwargs)

        self.validators.insert(0, station_key_validator)


class ModelField(fields.Field):
    def __init__(self, *args, **kwargs):
        self.model_klass = kwargs.pop('model', models.Model)
        super(ModelField, self).__init__(*args, **kwargs)

    def _deserialize(self, value, attr, data):
        # raise находится здесь, чтобы удобно отдавать reason клиенту
        if isinstance(value, Exception):
            raise value

        if not isinstance(value, self.model_klass):
            msg = u"%s is not instanse of %s" % (
                unicode(value), unicode(self.model_klass)
            )
            raise ValidationError(msg)

        return value


class DictNestedField(fields.Field):
    """ При необходимости доработать добавив все обычные параметры филдов """

    def __init__(self, nested):
        self.nested = nested
        self.__schema = None

        super(DictNestedField, self).__init__()

    @property
    def schema(self):
        if not self.__schema:
            self.__schema = self.nested()

        return self.__schema

    def _serialize(self, nested_list, attr, obj):
        schema = self.schema

        return {
            unicode(key): schema.dump(nested_obj).data
            for key, nested_obj in nested_list.items()
        }


class ListToDictField(fields.Nested):
    """ Превращает список в словарь этих же объектов с ключами из 'key' """

    def __init__(self, nested, key='id', **kwargs):
        self.key = key
        kwargs['many'] = True
        if 'only' in kwargs and key not in kwargs['only']:
            kwargs['only'].append(key)
        super(ListToDictField, self).__init__(nested, **kwargs)

    def _serialize(self, nested_obj, attr, obj):
        ret = super(ListToDictField, self)._serialize(nested_obj, attr, obj)
        try:
            return {unicode(val[self.key]): val for val in ret}
        except KeyError:
            msg = 'No key field "%s"' % self.key
            return ApiException(msg).dump()


class DatetimeAware(fields.Field):
    def _serialize(self, value, attr, obj):
        if not value:
            return None

        return {
            'local': value.strftime("%Y-%m-%dT%H:%M:%S"),
            'offset': timedelta2minutes(value.utcoffset()),
        }


class InputNested(fields.Nested):
    """ как обычный Nested, только позволяет прокидывать параметры из запроса """

    @property
    def schema(self):
        # Прокидываем параметры запроса только если они есть
        context = getattr(self.parent, 'context', {})
        parent_input_fields = context.get('fields', None)
        if self.name not in parent_input_fields:
            return super(InputNested, self).schema

        field_input_data = parent_input_fields.get(self.name, {})
        if field_input_data.get('params'):
            raise Exception('Params for nested fields not allowed')

        # Ставим only из запроса
        input_fields = field_input_data.get('fields', [])
        input_fields_dict = process_fields(self.nested, input_fields)
        if input_fields:
            self.only = input_fields_dict.keys()

        schema = super(InputNested, self).schema

        # Nested field прокидывает контекст схемы
        # Нам же нужен обработанный контекст филда
        if field_input_data:
            params = field_input_data.get('params', {})

            schema.context = {
                'params': params,
                'fields': input_fields_dict
            }

        return schema
