# -*- encoding: utf-8 -*-
from __future__ import absolute_import

from datetime import datetime, time, timedelta

import flask
from django.conf import settings
from marshmallow import Schema, fields, validates_schema, ValidationError

from travel.avia.library.python.avia_data.models import BestOffers
from travel.avia.library.python.common.models.geo import Settlement
from travel.avia.library.python.common.models.schedule import RThread
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.utils.date import get_local_time

from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.fields import ModelField, SettlementKey
from travel.avia.backend.main.lib.rates import convert_tariff, fetch_rates_info
from travel.avia.backend.main.lib.prices import AviaPrice

from travel.avia.backend.main.api_types.reference import ReferenceSchema, VariantSchema
from travel.avia.backend.main.api_types.settlement import get_settlement


# Этот модуль будет сильно отрефакторин когда будет делаться ручка update

def get_points_from_params(params):
    from_point = params.get('from_point')
    to_point = params.get('to_point')

    if not from_point:
        from_id = params.get('from_id')
        if not from_id and params.get('from_key'):
            from_id = params.get('from_key')[1:]
        from_point = get_settlement(from_id)

    if not to_point:
        to_id = params.get('to_id')
        if not to_id and params.get('to_key'):
            to_id = params.get('to_key')[1:]
        to_point = get_settlement(to_id)

    return from_point, to_point


class BestOffersParams(Schema):
    from_id = fields.Int()
    to_id = fields.Int()

    from_key = SettlementKey()
    to_key = SettlementKey()

    from_point = ModelField(model=Settlement)
    to_point = ModelField(model=Settlement)

    @validates_schema
    def not_empty(self, data):
        if all(not data.get(i) for i in ['from_id', 'from_key', 'from_point']):
            raise ValidationError('from_id or from_key required', 'from_id')

        if all(not data.get(i) for i in ['to_id', 'to_key', 'to_point']):
            raise ValidationError('to_id or to_key required', 'to_id')


def _get_thread(route, dt):
    db_threads = RThread.objects.filter(
        number=route, t_type=TransportType.PLANE_ID, hidden=False, type=1
    )

    for thread in db_threads:
        if thread.runs_at(dt):
            return thread


def _process_route(route, last_arrival, times=0):
    # На случай если данные в бд не верные - выходим пораньше
    # Двух итераций при нормальных данных должно всегда хватать
    if times > 2:
        return None

    last_arrival_date = last_arrival.date()
    thread = _get_thread(route, last_arrival_date)

    if not thread:
        return None

    rstation_from = thread.display_path[0]
    rstation_to = thread.display_path[len(thread.display_path) - 1]

    station_from = rstation_from.station
    station_to = rstation_to.station

    start_date = thread.first_run(last_arrival_date)
    naive_start_dt = datetime.combine(start_date, thread.tz_start_time)

    departure = rstation_from.get_loc_departure_dt(naive_start_dt)

    if not departure:
        return None

    if last_arrival and last_arrival > departure:
        return _process_route(route, last_arrival + timedelta(days=1), times + 1)

    arrival = rstation_to.get_loc_arrival_dt(naive_start_dt)

    if not arrival:
        return None

    stations = [station_from, station_to]

    settlements = []
    if station_from.settlement:
        settlements.append(station_from.settlement)

    if station_to.settlement:
        settlements.append(station_to.settlement)

    if thread.company:
        company = thread.company
    else:
        return None

    flight_key = "%s-%s-%s-%s-%s-%s" % (
        route,
        thread.company and thread.company.id,
        station_from and station_from.id,
        station_to and station_to.id,
        departure and departure.strftime('%m%d%H%M'),
        arrival and arrival.strftime('%m%d%H%M')
    )

    flight = {
        'number': route,
        'company': thread.company and thread.company.id,
        'from_': station_from and station_from.id,
        'to': station_to and station_to.id,
        'departure': departure,
        'arrival': arrival,
        'key': flight_key,
    }

    return flight_key, flight, stations, settlements, company, arrival


def _process_routes(routes, last_arrival, reference, lists):
    flight_keys = []

    for route in routes:
        result = _process_route(route, last_arrival)
        if not result:
            return None

        flight_key, flight, stations, settlements, company, last_arrival = result
        flight_keys.append(flight_key)

        reference['flights'][flight_key] = flight
        lists['stations'] += stations
        lists['settlements'] += settlements
        lists['companies'].append(company)

    f_key = '-'.join(routes)
    reference['itineraries'][f_key] = flight_keys

    return f_key


def get_best_offers(from_point, to_point):
    national_version = flask.g.get('national_version')
    offers = BestOffers.objects.filter(
        national_version=national_version,
        direction="%s_%s" % (from_point.point_key, to_point.point_key)
    )

    if not offers:
        return None

    reference = {
        'flights': {},
        'itineraries': {},
        'companies': {},
        'stations': {},
        'settlements': {},
        'alliances': {},
        'partners': [{
            # Это нигде не используется, нужно чтобы не падала вёрстка
            'id': 'ozon',
            'code': 'ozon',
            'logo': None,
            'logo_svg': None,
            'title': "Ozon",
        }]
    }

    lists = {
        'companies': [],
        'stations': [],
        'settlements': [],
    }

    variants = []

    for offer in offers:
        forward_routes = offer.forward_route.split(',')
        backward_routes = offer.backward_route.split(',') if offer.backward_route else []

        forward_key = _process_routes(
            forward_routes,
            get_local_time(from_point, datetime.combine(offer.forward_date, time(0, 0))),
            reference, lists
        )
        backward_key = None

        if backward_routes:
            backward_key = _process_routes(
                backward_routes,
                get_local_time(from_point,
                               datetime.combine(offer.backward_date, time(0, 0))),
                reference, lists
            )

        if forward_key:
            backward_key = None
            rates = fetch_rates_info(national_version)
            country_base = settings.AVIA_NATIONAL_CURRENCIES.get(national_version, 'EUR')

            tariff = AviaPrice(offer.price, offer.currency.code, offer.currency.iso_code, roughly=True)
            tariff = convert_tariff(tariff, rates, country_base)

            if tariff is None:
                continue

            variants.append({
                'forward': forward_key,
                'backward': backward_key,
                'partner': 'ozon',
                'tariff': tariff
            })

    reference['flights'] = reference['flights'].values()
    reference['settlements'] = set(lists['settlements'])
    reference['stations'] = set(lists['stations'])
    reference['alliances'] = set(lists['companies'])
    reference['companies'] = set(lists['companies'])

    if not variants:
        return None

    return {
        'reference': reference,
        'variants': variants,
    }


class BestOfferSchema(TypeSchema):
    reference = fields.Nested(ReferenceSchema)
    variants = fields.Nested(VariantSchema, many=True)


class BestOffersHandler(ApiHandler):
    PARAMS_SCHEMA = BestOffersParams
    TYPE_SCHEMA = BestOfferSchema

    def process(self, params, fields):
        from_point, to_point = get_points_from_params(params)

        return get_best_offers(from_point, to_point)
