# -*- encoding: utf-8 -*-
from __future__ import absolute_import

from django.conf import settings
from marshmallow import Schema, fields

from travel.avia.library.python.common.models.geo import Settlement

from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api_types.settlement import SettlementSchema


class ClientSettlementParams(Schema):
    geo_id = fields.Integer()
    root_domain = fields.String()


class ClientSettlementHandler(ApiHandler):
    PARAMS_SCHEMA = ClientSettlementParams
    TYPE_SCHEMA = SettlementSchema

    def preprocess_fields(self, fields):
        if not fields:
            fields = ['id', 'title', 'code']

        return fields

    def _get_settlement_by_geo_id(self, geo_id):
        try:
            return Settlement.objects.get(_geo_id=geo_id)
        except Settlement.DoesNotExist:
            pass

    def _get_default_city(self, root_domain):
        return Settlement.get_default_for_root_domain(root_domain)

    def _get_city_domain(self, root_domain, city):
        if root_domain == 'com':
            return 'com'

        try:
            detected_country_domain = city.country.domain_zone or 'com'
        except AttributeError:
            detected_country_domain = 'com'

        # Авиабилеты не имеют домена kz, by и тп
        # Поэтому срабатывает не нужная замен города пользователя для ajax запросов
        allowed_domains = getattr(settings, 'ALLOWED_DOMAINS', None)
        if allowed_domains and detected_country_domain not in allowed_domains:
            detected_country_domain = 'com'

        return detected_country_domain

    def process(self, params, fields):
        geo_id = params.get('geo_id')
        root_domain = params.get('root_domain')

        client_city_real = self._get_settlement_by_geo_id(geo_id)
        default_city = self._get_default_city(root_domain)
        client_city = client_city_real or default_city
        client_city_domain = self._get_city_domain(root_domain, client_city)

        if root_domain != client_city_domain:
            client_city = default_city

        return client_city
