# -*- encoding: utf-8 -*-
from __future__ import absolute_import

import flask
from unidecode import unidecode
from marshmallow import Schema, fields
from django.conf import settings
from django.utils.translation import get_language

from travel.avia.library.python.common.models.geo import Country

from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.fields import ModelField, Related
from travel.avia.backend.main.api.register import get_handler
from travel.avia.backend.main.lib.geo import get_capital


def get_country(code):
    if unicode(code).isdigit():  # нам передали id
        return Country.objects.get(pk=code)
    else:
        return Country.objects.get(code=code)


def get_url_title(s):
    lang, _ = settings.DOMAIN_LANGUAGE_MAP.get(flask.g.national_version, 'en')
    return unidecode(s.L_title(lang=lang)).replace("'", '')


class CountryParams(Schema):
    id = fields.Integer()
    country = ModelField(model=Country, allow_none=True)


class CountrySchema(TypeSchema):
    id = fields.Int()
    geo_id = fields.Int(attribute='_geo_id')
    title = fields.Str(attribute='L_title')
    url_title = fields.Function(get_url_title)
    code = fields.Str()
    key = fields.Str(attribute='point_key')
    capital = Related(handler=get_handler('settlement'), params={
        'settlement': lambda country, context: get_capital(country, flask.g.national_version),
    })
    phrase_from = fields.Method('get_phrase_from')
    phrase_to = fields.Method('get_phrase_to')

    def get_phrase_to(self, s):
        lang = get_language()

        if lang not in ('ru', 'uk'):
            return s.L_title()

        title = s.L_title(case='accusative', fallback=False)

        if not title:
            return s.L_title()

        return title

    def get_phrase_from(self, s):
        lang = get_language()

        if lang not in ('ru', 'uk'):
            return s.L_title()

        title = s.L_title(case='genitive', fallback=False)

        if not title:
            return s.L_title()

        return title


class CountryHandler(ApiHandler):
    PARAMS_SCHEMA = CountryParams
    TYPE_SCHEMA = CountrySchema

    def preprocess_fields(self, fields):
        if not fields:
            return ['id', 'title']

        return fields

    def process(self, params, fields):
        if params.get('country'):
            return params.get('country')

        country_code = params.get('id') or params.get('code')

        if not country_code and params.get('key'):
            key = params.get('key')
            country_code = key[1:]

        if not country_code:
            return None

        return get_country(country_code)
