# -*- encoding: utf-8 -*-
from __future__ import absolute_import

import flask
from django.conf import settings
from marshmallow import fields, validates_schema, ValidationError

from travel.avia.library.python.avia_data.models import AviaDirectionNational, AviaSettlementNational
from travel.avia.library.python.common.models.geo import Settlement, Country

from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.fields import ModelField, SettlementKey, CountryKey
from travel.avia.backend.main.api_types.country import get_country as get_country_by_id
from travel.avia.backend.main.lib.directions import SettlementDirectionFinder, apply_passengers
from travel.avia.backend.main.lib.geo import has_airport
from travel.avia.backend.main.api_types.offers import OfferSchema
from travel.avia.backend.main.api_types.reference import FormBaseParams
from travel.avia.backend.main.api_types.settlement import get_settlement, fill_related_fields as fill_settlement_related_fields


def get_settlement_and_country(params):
    from_point = params.get('from_point')
    to_point = params.get('to_point')

    if not from_point:
        from_id = params.get('from_id')
        if not from_id and params.get('from_key'):
            from_id = params.get('from_key')[1:]
        from_point = get_settlement(from_id)

    if not to_point:
        to_id = params.get('to_id')
        if not to_id and params.get('to_key'):
            to_id = params.get('to_key')[1:]
        to_point = get_country_by_id(to_id)

    return from_point, to_point


def get_directions(from_city, to_country, when, return_date, params):
    national_version = flask.g.get('national_version', None)

    settlements = AviaDirectionNational.objects.filter(
        departure_settlement=from_city,
        arrival_settlement__country=to_country,
        national_version=national_version
    ).exclude(arrival_settlement=from_city)

    # Выкидываем Симферополь из России для не рус версии
    if to_country.id == settings.RUSSIA_GEO_ID and national_version != 'ru':
        settlements = settlements.exclude(arrival_settlement=146)

    # Выкидываем Эрджан из Кипра в тур версии
    if to_country.id == 20574 and national_version == 'tr':
        settlements = settlements.exclude(arrival_settlement=40775)

    settlements = settlements.values_list(
        'arrival_settlement', 'popularity'
    )

    settlement_ids = [s[0] for s in settlements]
    popularity_dict = {
        s[0]: s[1]
        for s in settlements
    }

    if len(settlement_ids) < 20:
        avia_settlements = AviaSettlementNational.objects.filter(
            settlement__country=to_country
        ).exclude(
            settlement=from_city,
            settlement_id__in=settlement_ids
        ).order_by(
            '-popularity'
        )[:10]

        settlement_ids += [s.settlement_id for s in avia_settlements if s.settlement_id]

    # Добавим Симферополь Украине для уа
    if (
        to_country.id == settings.UKRAINE_GEO_ID
        and national_version == 'ua'
        and from_city.id != 146
    ):
        settlement_ids = list(settlement_ids) + [146]

    # Добавляем Эрджан Турции в тур версии
    if (
        to_country.id == settings.TURKEY_GEO_ID
        and national_version == 'tr'
        and from_city.id != 40775
    ):
        settlement_ids = list(settlement_ids) + [40775]

    if params['only_with_airports']:
        settlement_ids = list(filter(
            lambda s_id: has_airport(s_id),
            settlement_ids
        ))

    finder = SettlementDirectionFinder(
        from_city, limit=10000, pad=0,
        when=when,
        return_date=return_date,
        arrival_ids=settlement_ids
    )

    directions = finder.find()

    directions.sort(key=lambda d: d.city_to.majority_id)
    directions.sort(key=lambda d: -popularity_dict.get(d.city_to.id, 0))

    return apply_passengers(params, directions)


class CountryDirectionsParams(FormBaseParams):
    from_id = fields.Int()
    to_id = fields.Int()

    from_key = SettlementKey()
    to_key = CountryKey()

    from_point = ModelField(model=Settlement)
    to_point = ModelField(model=Country)

    only_with_airports = fields.Bool(required=False, missing=False)

    @validates_schema
    def not_empty(self, data):
        if all(not data.get(i) for i in ['from_id', 'from_key', 'from_point']):
            raise ValidationError('from_id or from_key required', 'from_id')

        if all(not data.get(i) for i in ['to_id', 'to_key', 'to_point']):
            raise ValidationError('to_id or to_key required', 'to_id')


class CountryDirectionsHandler(ApiHandler):
    PARAMS_SCHEMA = CountryDirectionsParams
    TYPE_SCHEMA = OfferSchema
    MULTI = True

    def preprocess_fields(self, fields):
        if not fields:
            return [
                'from_city', 'to_city', 'direct_price', 'transfers_price',
                'date_forward', 'date_backward'
            ]

        return fields

    def process(self, params, fields):
        from_point, to_point = get_settlement_and_country(params)
        offers = get_directions(from_city=from_point,
                                to_country=to_point,
                                when=params.get('when'),
                                return_date=params.get('return_date'),
                                params=params)

        sort_by_price = fields.get('directPrice') or fields.get('transfersPrice')
        if sort_by_price:
            offers.sort(key=lambda x: (
                -int(bool(x.indirect_price or x.direct_price)))
            )

        fill_settlement_related_fields([o.city_to for o in offers], fill_capital=True)

        return offers
