# -*- encoding: utf-8 -*-
from __future__ import absolute_import

from flask import request
from marshmallow import Schema, fields
from datetime import date, timedelta

from travel.avia.library.python.avia_data.models import SettlementBigImage
from travel.avia.library.python.common.utils.geo import great_circle_distance_km, GreatCircleCalculationError

from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.fields import Related, SettlementKey
from travel.avia.backend.main.api.register import get_handler
from travel.avia.backend.main.api_types.settlement import get_settlement
from travel.avia.backend.main.api_types.price import MinPricesSchema, get_prices_from_db, get_prices_from_cache


class DirectionParams(Schema):
    # Пока принимаем только города, вроде ап нигде не нужны
    from_key = SettlementKey(required=True)
    to_key = SettlementKey(required=True)
    when = fields.Date()

    # даты внутри которых нужно производить поиск
    left_date = fields.Date()
    right_date = fields.Date()


class DirectionImageSchema(TypeSchema):
    url2 = fields.Str()
    url = fields.Str()
    big_url = fields.Str()
    small_url = fields.Str()
    large_url = fields.Str()
    header_url = fields.Str()


class DirectionSchema(TypeSchema):
    min_price = fields.Nested(MinPricesSchema)
    min_price_date = fields.Str()
    distance = fields.Int()  # km

    image = fields.Nested(DirectionImageSchema)

    similar = Related(
        handler=get_handler('similarDirections'),
        params={
            'from_point': 'from_point',
            'to_point': 'to_point',
        },
        extra_params={
            'only_with_airports': True,
        },
    )

    near = Related(handler=get_handler('nearDirections'), params={
        'from_point': 'from_point',
        'to_point': 'to_point'
    })


def get_image(city_to):
    try:
        return SettlementBigImage.objects.get(settlement_id=city_to.id)
    except SettlementBigImage.DoesNotExist:
        return SettlementBigImage.get_default_image(request)


class DirectionHandler(ApiHandler):
    PARAMS_SCHEMA = DirectionParams
    TYPE_SCHEMA = DirectionSchema

    def process(self, params, fields):
        from_point = get_settlement(params.get('from_key')[1:])
        to_point = get_settlement(params.get('to_key')[1:])
        when = params.get('when', None)

        result = {
            'from_point': from_point,
            'to_point': to_point,
        }

        if 'min_price' in fields:
            # ищем на неделю вперёд и только в одну сторону
            # при необходимости это можно будет передать параметрами
            if when:
                left_date = right_date = when
            else:
                left_date = params.get('left_date', date.today())
                right_date = params.get('right_date', left_date + timedelta(days=30))

            # ищем простым способом, т.к. направление известно и всего 30 дней
            days_prices = get_prices_from_db(
                from_point, to_point, left_date, right_date,
                delta=None, passengers_key='1_0_0', force_one_passenger=False
            )
            days_cached_prices = get_prices_from_cache(
                from_point, to_point, left_date, right_date,
                delta=None, passengers_key='1_0_0', force_one_passenger=False
            )
            days_prices.update(days_cached_prices)
            if days_prices:
                def prices_sort_key(p):
                    if 'direct' in p and 'transfers' in p:
                        return min(p['direct'], p['transfers'])
                    elif 'direct' in p:
                        return p['direct']
                    elif 'transfers' in p:
                        return p['transfers']
                    return None

                sorted_days_prices = sorted(days_prices.items(), key=lambda x: prices_sort_key(x[1]))
                min_day_price = sorted_days_prices[0]
                result['min_price_date'] = min_day_price[0]
                result['min_price'] = min_day_price[1]

        if 'distance' in fields:
            try:
                distance = round(great_circle_distance_km(from_point, to_point))
            except GreatCircleCalculationError:
                distance = None

            result['distance'] = distance

        if 'image' in fields:
            result['image'] = get_image(to_point)

        return result
