# -*- encoding: utf-8 -*-
from __future__ import absolute_import

import logging
from locale import strcoll

import flask
from flask import g
from marshmallow import Schema, fields
from django.utils.translation import get_language
from django.conf import settings

from travel.avia.library.python.common.models.currency import Currency
from travel.avia.library.python.common.models.geo import Settlement
from travel.avia.library.python.avia_data.models import AviaDirectionNational, AviaSettlementNational, MinPrice

from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.fields import ModelField, SettlementKey, InputNested
from travel.avia.backend.main.lib.directions import DirectionRawSQLFinder
from travel.avia.backend.main.lib.prices import DirectionBasePrice, PrefetchedDirectionPrice
from .offers import OfferSchema
from .settlement import (get_settlement, get_country,
                         get_iata_code, fill_related_fields as fill_settlement_related_fields)

log = logging.getLogger(__name__)

TURKISH_CHAR_CODES = [199, 231, 286, 287, 304, 305, 214, 246, 350, 351, 220, 252]


def incorrect_settlement_transctiption(lang, sid):
    log.warning('Transcription is not correct for {lang} settlement {sid}'.format(sid=sid, lang=lang))


def get_settlement_groups(settlement_ids):
    settlement_groups = {}
    settlements = list(Settlement.objects.filter(id__in=settlement_ids))

    for settlement in settlements:
        letter = settlement.L_title().lower()[0]
        sid = settlement.id
        letter_code = ord(letter)

        lang = get_language()

        if lang in ['ru', 'uk', 'tr']:
            # Кириллица начинается с 1024
            if lang in ['ru', 'uk'] and letter_code < 1024:
                incorrect_settlement_transctiption(lang, sid)
                continue

            if lang == 'tr' and (
                    letter_code > 128 and letter_code not in TURKISH_CHAR_CODES):
                incorrect_settlement_transctiption(lang, sid)
                continue
        else:
            # RASPTICKETS-4492
            # Символы 128..256 входят в группу latin-1.
            if letter_code > 256:
                incorrect_settlement_transctiption(lang, sid)
                continue

        ids = settlement_groups.setdefault(letter, [])
        ids.append(sid)

    return settlement_groups


class LetterDirectionFinder(DirectionRawSQLFinder):
    def get_letter_arrival_settlement(self):
        arrival_ids = list(AviaDirectionNational.objects.filter(
            departure_settlement=self.city_from,
            national_version=self.national_version
        ).exclude(
            arrival_settlement=self.city_from
        ).order_by('arrival_settlement').values_list(
            'arrival_settlement', flat=True
        ).distinct())

        return get_settlement_groups(arrival_ids)

    def find(self, letters):
        settlements = self.get_letter_arrival_settlement()

        arrival_ids = []
        for letter, s_ids in settlements.items():
            if letter in letters:
                arrival_ids += s_ids

        directions = self._get_directions(self.city_from, arrival_ids)

        no_price_directions = []
        found_ids = set([d.city_to.id for d in directions])

        settlement_ids = set([sid for sid in arrival_ids if sid not in found_ids])
        settlements = list(Settlement.objects.filter(id__in=settlement_ids))

        for settlement in settlements:
            no_price_directions.append(DirectionBasePrice(
                self.city_from, settlement,
                national_version=self.national_version
            ))

        directions = self._memcache_update(directions)

        directions = directions + no_price_directions

        return sorted(directions, key=lambda d: d.city_to.L_title())[:self.limit]


class CheapestDirectionFinder(DirectionRawSQLFinder):
    def _get_directions(self, city_from):
        date_where = self._get_date_where()

        # 1. Вытащим города с минимальными ценами подходящие под фильтры.
        #    В результате будет связка откуда+куда+цена
        inner_sql = """
SELECT
departure_settlement_id, arrival_settlement_id, national_version, passengers,
min(price) AS minprice
FROM www_minprice
WHERE
passengers='1_0_0' AND
national_version=%%s AND
%s AND
departure_settlement_id=%%s
group by arrival_settlement_id
order by minprice
limit 36
        """ % date_where
        # 2. По связке вытаскиваем все записи где подходящие под откуда+куда+цена
        #    группированные по датам. Обычно их получается не много (сотни)
        grouped_sql = """
SELECT m.*
FROM (%s) as x INNER JOIN www_minprice AS m ON
m.departure_settlement_id = x.departure_settlement_id AND
m.arrival_settlement_id = x.arrival_settlement_id AND
m.national_version = x.national_version AND
m.passengers = x.passengers AND
m.price = x.minprice AND
%s
ORDER BY m.date_forward, m.date_backward
        """ % (
            inner_sql,
            # снова нужен where, т.к. откуда+куда+цена могут быть на другие даты
            date_where.replace(
                'date_forward', 'm.date_forward'
            ).replace(
                'date_backward', 'm.date_backward'
            )
        )

        # 3. Вытаскиваем запись с самой ранней датой через subquery (это хак)
        raw_sql = """
select u.* from (%s) as u
group by u.departure_settlement_id, u.arrival_settlement_id, u.price
order by u.price;
        """ % grouped_sql

        min_prices_part1 = list(MinPrice.objects.raw(raw_sql, [
            self.national_version,
            city_from.id
        ]))

        # Какой-нибудь Сидней или США, где вообще нет цен
        if not min_prices_part1:
            return []

        # Достаем прямые/пересадочные цены на эти же даты
        raw_sql2 = []
        for i in min_prices_part1:
            raw_sql2.append("""
SELECT * FROM www_minprice
WHERE
passengers='1_0_0' AND
national_version='%s' AND
date_forward='%s' AND
date_backward%s AND
departure_settlement_id=%d AND
arrival_settlement_id=%d AND
direct_flight=%d
            """ % (
                self.national_version,
                i.date_forward.isoformat(),
                "='%s'" % i.date_backward.isoformat() if i.date_backward else ' IS NULL',
                city_from.id,
                i.arrival_settlement_id,
                0 if i.direct_flight else 1
            ))

        min_prices_part2 = MinPrice.objects.raw(' UNION ALL '.join(raw_sql2) + ';')
        min_prices_part2_dict = self._min_prices_to_dict(min_prices_part2)

        currency_ids = [p.currency_id for p in min_prices_part1]
        currency_ids += [p.currency_id for p in min_prices_part2_dict.values()]
        currencies = {c.id: c for c in list(Currency.objects.filter(id__in=set(currency_ids)))}

        settlement_ids = set([p.arrival_settlement_id for p in min_prices_part1])
        settlements = {
            s.id: s for s in list(Settlement.objects.filter(id__in=settlement_ids))
        }

        # Создаём объекты направлений с обоими ценами
        directions = []
        for min_price in min_prices_part1:
            key = "%d_%d" % (min_price.departure_settlement_id, min_price.arrival_settlement_id)
            min_price2 = min_prices_part2_dict.get(key)
            min_price.currency = currencies[min_price.currency_id]
            if min_price2:
                min_price2.currency = currencies[min_price2.currency_id]

            direction = PrefetchedDirectionPrice(
                city_from, settlements[min_price.arrival_settlement_id],
                min_price=min_price, min_price2=min_price2,
                allow_roughly=True,
                national_version=self.national_version
            )

            directions.append(direction)

        return directions

    def find(self):
        return self._get_directions(self.city_from)


def get_popular_cities(params, fields, national_version):
    result = {}

    settlement_ids = list(AviaSettlementNational.objects.filter(
        arrival=False,
        national_version=g.get('national_version')
    ).order_by('-popularity').values_list('settlement', flat=True))

    # RASPTICKETS-3585 Хак для Украины
    if national_version == 'ua':
        ua_settlement_ids = list(AviaSettlementNational.objects.filter(
            arrival=False, settlement__country_id=settings.UKRAINE_GEO_ID,
            national_version=national_version
        ).order_by('-popularity').values_list('settlement', flat=True))
        ua_settlement_set = set(ua_settlement_ids)
        settlement_ids = ua_settlement_ids + [x for x in settlement_ids if x not in ua_settlement_set]

    settlement_groups = get_settlement_groups(settlement_ids)

    all_letters = sorted(settlement_groups.keys(), cmp=strcoll)

    settlement_ids = settlement_ids[:50]
    settlement_id_to_pos = {x: i for i, x in enumerate(settlement_ids)}
    settlements = list(Settlement.objects.filter(id__in=settlement_ids))

    settlements.sort(key=lambda x: settlement_id_to_pos[x.id])

    for settlement in settlements:
        settlement.iata = get_iata_code(settlement)
        settlement.country = get_country(settlement)

    result['letters'] = [{
        'name': l.lower(),
        'count': len(settlement_groups[l])
    } for l in all_letters]

    result['offers'] = [{
        'city_to': s
    } for s in settlements]

    return result


# Дальше всё что относится к хендлерам

class OffersFilters(Schema):
    letters = fields.List(fields.Str)


class GeoCitiesParams(Schema):
    from_id = fields.Int()
    from_key = SettlementKey()
    from_point = ModelField(model=Settlement)

    offers_filters = fields.Nested(OffersFilters)


class LetterSchema(TypeSchema):
    name = fields.Str()
    count = fields.Int()


class GeoCitiesSchema(TypeSchema):
    # Список букв на которые начинаются города
    letters = InputNested(LetterSchema, many=True)
    # Направления
    offers = InputNested(OfferSchema, many=True)


class GeoCitiesHandler(ApiHandler):
    PARAMS_SCHEMA = GeoCitiesParams
    TYPE_SCHEMA = GeoCitiesSchema

    def process(self, params, fields):
        from_point = params.get('from_point')

        if not from_point:
            from_id = params.get('from_id')
            if not from_id and params.get('from_key'):
                from_id = params.get('from_key')[1:]
            if from_id:
                from_point = get_settlement(from_id)

        if not from_point:
            return get_popular_cities(params, fields, flask.g.national_version)

        result = {}

        finder = LetterDirectionFinder(from_point, limit=1000)

        if 'letters' in fields:
            settlement_groups = finder.get_letter_arrival_settlement()
            all_letters = sorted(settlement_groups.keys(), cmp=strcoll)

            result['letters'] = [{
                'name': l,
                'count': len(settlement_groups[l])
            } for l in all_letters]

        if 'offers' in fields:
            result['offers'] = None

            offers_filters = params.get('offers_filters', {})
            if offers_filters and 'letters' in offers_filters:
                result['offers'] = finder.find(offers_filters.get('letters'))
            else:  # По умолчанию возвращаем дешёвые
                result['offers'] = CheapestDirectionFinder(from_point, limit=50).find()

            fill_settlement_related_fields([o.city_to for o in result['offers']])

        return result
