# -*- encoding: utf-8 -*-
from __future__ import absolute_import

from locale import strcoll

from flask import g
from marshmallow import Schema, fields, validates_schema, ValidationError

from travel.avia.library.python.common.models.geo import Country, Settlement
from travel.avia.library.python.avia_data.models import AviaDirectionNational

from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.fields import ModelField, SettlementKey
from .country import CountrySchema
from .settlement import get_settlement


def get_counties(city_from, national_version):
    country_ids = AviaDirectionNational.objects.filter(
        departure_settlement=city_from,
        national_version=national_version
    ).order_by('arrival_settlement__country').values_list(
        'arrival_settlement__country', flat=True
    ).distinct()

    return sorted(
        [Country.objects.get(id=cid) for cid in country_ids],
        key=lambda x: x.L_title(), cmp=strcoll
    )


class GeoCountriesParams(Schema):
    from_id = fields.Int()
    from_key = SettlementKey()
    from_point = ModelField(model=Settlement)

    @validates_schema
    def not_empty(self, data):
        if all(not data.get(i) for i in ['from_id', 'from_key', 'from_point']):
            raise ValidationError('from_id or from_key required', 'from_id')


class GeoCountriesHandler(ApiHandler):
    PARAMS_SCHEMA = GeoCountriesParams
    TYPE_SCHEMA = CountrySchema
    MULTI = True

    def process(self, params, fields):
        from_point = params.get('from_point')

        if not from_point:
            from_id = params.get('from_id')
            if not from_id and params.get('from_key'):
                from_id = params.get('from_key')[1:]
            from_point = get_settlement(from_id)

        return get_counties(from_point, g.get('national_version'))
