# -*- encoding: utf-8 -*-
from __future__ import absolute_import

import flask
from marshmallow import Schema, fields
from django.core.exceptions import ObjectDoesNotExist

from travel.avia.library.python.common.models.geo import Country, Settlement, Station
from travel.avia.library.python.geosearch.views.point import PointSearch, TooShortError, InvalidPointKey, NoPointsError, StopWordError
from travel.avia.library.python.geosearch.views.pointtopoint import process_points_lists, SamePointError

from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.fields import InputNested, Related
from travel.avia.backend.main.api.register import get_handler
from travel.avia.backend.main.lib.geo import has_own_airport, get_airport_city, get_country_cities, get_settlement_by_geoid, \
    get_point_by_key
from .settlement import get_settlement, get_country as get_settlement_country
from .station import get_country as get_station_country
from .country import get_country


def get_search_city(city, national_version):
    # Город пользователя может быть без аэропорта, тогда предзаполнение будет не верное
    if not has_own_airport(city):
        return get_airport_city(city, national_version)

    return city


def _get_point_list(id, name):
    try:
        point_list = PointSearch.find_point(name, 'plane', id)
    except TooShortError:
        return None, 'too_short'
    except InvalidPointKey:
        return None, 'invalid_key'
    except StopWordError:
        return None, 'stop_word'
    except (ObjectDoesNotExist, NoPointsError):
        return None, 'does_not_exist'

    return point_list, None


def _get_point_from_pointlist(point_list):
    point = None

    if point_list is not None:
        if point_list.point:
            point = point_list.point
        elif len(point_list.variants):
            # Берём первый вариант, благодаря сортировке там лежит самый подходящий
            point = point_list.variants[0]

    return point


def _safe_get_settlement(code, search_city=None):
    if code == '-me':
        return search_city
    try:
        return get_settlement(code)
    except:
        return None


def _safe_get_country(code, default=None):
    try:
        return get_country(code)
    except:
        return default


def _get_direction_fields(direction, fields):
    geo_fields = ['station', 'settlement', 'country', 'adjustments']
    return set([i for i in geo_fields if '%s_%s' % (direction, i) in fields])


def find_from_point(search_city, from_params):
    from_code, from_id, from_name = [
        from_params.get(k) for k in ['from_code', 'from_id', 'from_name']
    ]

    if not from_id and not from_name:
        if from_code:
            return _safe_get_settlement(from_code, search_city), None, None
        return search_city, None, None

    point_list, error = _get_point_list(from_id, from_name)

    if error:
        return None, error, None

    point = _get_point_from_pointlist(point_list)
    adjustments = None

    # Если в точке страна, то нужно подсказать города с аэропортами
    if isinstance(point, Country):
        adjustments = get_country_cities(point)
        error = 'choose_airports'
    elif point_list and point_list.has_variants():
        adjustments = filter(
            lambda x: (
                x != point_list.point and not isinstance(x, Country)
            ),
            point_list.variants
        )

    return point, None, adjustments


def _get_settlement_by_code(point_code):
    point = None

    if point_code:
        point = _safe_get_settlement(point_code)

    return point


def _get_point_by_code(point_key, point_code, search_city, fields):
    point = None
    error = None

    # Если поиск от/до страны и указан код, то пропускаем этот блок
    if point_code and any([i in fields for i in ['settlement', 'station']]):
        if point_code == '-me':
            return search_city, None

        point = _get_settlement_by_code(point_code)

        if not point:
            try:
                point = Station.get_by_code('iata', point_code)
            except:
                pass

    if not point and point_key:
        try:
            point = get_point_by_key(point_key)
        except:
            pass

    return point, error


def find_points(search_city, from_params, to_params, fields):
    from_point = None
    to_point = None

    from_code, from_id, from_name = [
        from_params.get(k) for k in ['from_code', 'from_id', 'from_name']
    ]

    to_code, to_id, to_name = [
        to_params.get(k) for k in ['to_code', 'to_id', 'to_name']
    ]

    from_error = None
    to_error = None

    from_fields = _get_direction_fields('from', fields)
    to_fields = _get_direction_fields('to', fields)

    if from_id or from_code:
        from_point, from_error = _get_point_by_code(from_id, from_code, search_city, from_fields)

    if to_id or to_code:
        to_point, to_error = _get_point_by_code(to_id, to_code, search_city, to_fields)
        if not to_point and to_code:
            to_point = _safe_get_country(to_code)
            if to_point:
                to_error = None

    # Проверка на совпадения города вылета и прибытия
    if from_point and to_point:
        from_city = from_point
        to_city = to_point

        if from_point.id == to_point.id:
            from_error = 'same_points'

        if isinstance(from_point, Station):
            from_city = from_point.settlement
        if isinstance(to_point, Station):
            to_city = to_point.settlement

        # Станция может быть не привязана к городу
        try:
            if from_city.id == to_city.id:
                from_error = 'same_points'
        except:
            pass

    # Если к этому моменту уже есть обе точки - искать не нужно
    if from_point and to_point:
        return {
            'points': [from_point, to_point],
            'errors': [from_error, to_error],
            'adjustments': [None, None],
        }

    if from_point:
        from_id = from_point.point_key
        from_name = from_point.L_title()

    if to_point:
        to_id = to_point.point_key
        to_name = to_point.L_title()

    return _find_points_geosearch(from_id, from_name, to_id, to_name, search_city)


def _find_points_geosearch(from_id, from_name, to_id, to_name, search_city):
    from_point = None
    to_point = None

    from_adjustments = None
    to_adjustments = None

    point_list_from, from_error = _get_point_list(from_id, from_name)
    point_list_to, to_error = _get_point_list(to_id, to_name)

    # Обе точки валидные, уточняем
    if not from_error and not to_error:
        try:
            point_list_from, point_list_to = process_points_lists(
                point_list_from, point_list_to,
                search_city,
                disable_replace=True
            )
        except SamePointError:
            from_error = 'same_points'

    if not from_error and point_list_from:
        from_point = _get_point_from_pointlist(point_list_from)

        # Если в точке страна, то нужно подсказать города с аэропортами
        if isinstance(from_point, Country):
            from_adjustments = get_country_cities(from_point)
            from_error = 'choose_airports'
        elif point_list_from and point_list_from.has_variants():
            from_adjustments = filter(
                lambda x: (
                    x != point_list_from.point and not isinstance(x, Country)
                ),
                point_list_from.variants
            )

    if not to_error and point_list_to:
        to_point = _get_point_from_pointlist(point_list_to)

        # Если в точке страна, то нужно подсказать города с аэропортами
        if isinstance(to_point, Country):
            to_adjustments = get_country_cities(to_point)
            to_error = 'choose_airports'
        elif point_list_to and point_list_to.has_variants():
            to_adjustments = filter(
                lambda x: (
                    x != point_list_to.point and not isinstance(x, Country)
                ),
                point_list_to.variants
            )

    return {
        'points': [from_point, to_point],
        'errors': [from_error, to_error],
        'adjustments': [from_adjustments, to_adjustments],
    }


class GeoParams(Schema):
    geo_id = fields.Int()

    # в этом месте не нужно валидировать инпут,
    # т.к. его проверит geosearch
    from_id = fields.Str()
    to_id = fields.Str()

    from_name = fields.Str()
    to_name = fields.Str()

    from_code = fields.Str()
    to_code = fields.Str()


def get_handler_by_point_type(obj, params):
    point = params['point']
    del params['point']

    if isinstance(point, Settlement):
        params['settlement'] = point
        return get_handler('settlement')(obj)
    elif isinstance(point, Station):
        params['station'] = point
        return get_handler('station')(obj)
    elif isinstance(point, Exception):
        raise point
    else:
        raise Exception('wrong point type')


class AdjustmentsSchema(TypeSchema):
    id = fields.Int()
    key = fields.Str(attribute='point_key')
    title = fields.Str(attribute='L_title')
    country = Related(handler=get_handler('country'), params={'country': 'country'})
    code = fields.Function(lambda s: s.iata or s.sirena_id)
    ptype = fields.Function(lambda s: 'station' if isinstance(s, Station) else 'settlement')


class GeoSchema(TypeSchema):
    search_city = Related(handler=get_handler('settlement'), params={'settlement': 'search_city'})
    client_city = Related(handler=get_handler('settlement'), params={'settlement': 'client_city'})

    from_station = Related(handler=get_handler('station'), params={'station': 'from_station'})
    to_station = Related(handler=get_handler('station'), params={'station': 'to_station'})

    from_settlement = Related(
        handler=get_handler('settlement'), params={'settlement': 'from_settlement'})
    to_settlement = Related(
        handler=get_handler('settlement'), params={'settlement': 'to_settlement'})

    from_settlement_hidden = Related(
        handler=get_handler('settlement'), params={'settlement': 'from_settlement_hidden'})
    to_settlement_hidden = Related(
        handler=get_handler('settlement'), params={'settlement': 'to_settlement_hidden'})

    from_country = Related(handler=get_handler('country'), params={'country': 'from_country'})
    to_country = Related(handler=get_handler('country'), params={'country': 'to_country'})

    from_adjustments = InputNested(AdjustmentsSchema, many=True)
    to_adjustments = InputNested(AdjustmentsSchema, many=True)

    from_error = fields.Str()
    to_error = fields.Str()


class GeoLookupHandler(ApiHandler):
    PARAMS_SCHEMA = GeoParams
    TYPE_SCHEMA = GeoSchema

    def preprocess_fields(self, fields):
        if not fields:
            fields = ['client_city', 'search_city', 'from_settlement', 'from_error']

        return fields

    def _need_direction(self, direction, fields):
        geo_fields = ['station', 'settlement', 'country', 'adjustments', 'settlement_hidden']
        geo_fields = ['%s_%s' % (direction, i) for i in geo_fields]
        return any([i in fields for i in geo_fields])

    def _point_to_geo(self, point):
        station = None
        settlement = None
        country = None

        if isinstance(point, Settlement):
            settlement = point
            settlement.country = get_settlement_country(settlement)
            country = settlement.country

        elif isinstance(point, Station):
            station = point
            settlement = point.settlement
            if settlement:
                settlement.country = get_settlement_country(settlement)
            station.country = get_station_country(station) if station.country else None
            country = station.country or (
                settlement.country if settlement else None
            )

        elif isinstance(point, Country):
            country = point

        return station, settlement, country

    def process(self, params, fields):
        client_city = get_settlement_by_geoid(
            params.get('geo_id'), flask.g.get('national_version')
        )

        search_city = get_search_city(client_city, flask.g.get('national_version'))

        from_point = None
        to_point = None

        from_error = None
        to_error = None

        from_adjustments = None
        to_adjustments = None

        from_params = {
            'from_id': params.get('from_id'),
            'from_name': params.get('from_name'),
            'from_code': params.get('from_code'),
        }

        to_params = {
            'to_id': params.get('to_id'),
            'to_name': params.get('to_name'),
            'to_code': params.get('to_code'),
        }

        # когда нужны обе точки их можно искать только вместе
        # точку "куда" нельзя вычислять без "откуда"
        if self._need_direction('to', fields):
            result = find_points(search_city, from_params, to_params, fields)
            from_point, to_point = result['points']
            from_error, to_error = result['errors']
            from_adjustments, to_adjustments = result['adjustments']

        elif self._need_direction('from', fields) or any(from_params.values()):
            from_point, from_error, from_adjustments = find_from_point(search_city, from_params)

        from_station = None
        to_station = None
        from_settlement = None
        to_settlement = None
        from_country = None
        to_country = None

        if from_point:
            from_station, from_settlement, from_country = self._point_to_geo(from_point)

        if to_point:
            to_station, to_settlement, to_country = self._point_to_geo(to_point)

        from_settlement_hidden = None
        to_settlement_hidden = None
        if from_settlement and (from_settlement.hidden or not has_own_airport(from_settlement)):
            from_settlement_hidden = from_settlement
            from_settlement = get_search_city(from_settlement, flask.g.get('national_version'))

        if to_settlement and (to_settlement.hidden or not has_own_airport(to_settlement)):
            to_settlement_hidden = to_settlement
            to_settlement = get_search_city(to_settlement, flask.g.get('national_version'))

        # Если есть город "откуда" он должен ставится в search_city
        if from_settlement:
            search_city = from_settlement

        return {
            'client_city': client_city,
            'search_city': search_city,

            'from_station': from_station,
            'from_settlement': from_settlement,
            'from_settlement_hidden': from_settlement_hidden,
            'from_country': from_country,

            'to_station': to_station,
            'to_country': to_country,
            'to_settlement': to_settlement,
            'to_settlement_hidden': to_settlement_hidden,

            'from_error': from_error,
            'to_error': to_error,

            'from_adjustments': from_adjustments,
            'to_adjustments': to_adjustments,
        }
