# -*- encoding: utf-8 -*-
from __future__ import absolute_import

import logging

import flask
from django.conf import settings
from flask import g
from marshmallow import Schema, fields, validate, validates_schema, ValidationError

from travel.avia.library.python.avia_data.models import AviaDirectionNational, NearDirection
from travel.avia.library.python.common.models.geo import Settlement

from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.fields import ModelField, SettlementKey
from travel.avia.backend.main.lib.prices import DirectionPrice
from travel.avia.backend.repository.s2s import geo_relations_repository
from travel.avia.backend.repository.settlement import settlement_geo_index
from .offers import OfferSchema
from .price import get_passengers_key
from .reference import FormBaseParams
from .settlement import get_settlement

log = logging.getLogger(__name__)


def get_settlements_from_params(params):
    from_point = params.get('from_point')
    to_point = params.get('to_point')

    if not from_point:
        from_id = params.get('from_id')
        if not from_id and params.get('from_key'):
            from_id = params.get('from_key')[1:]
        from_point = get_settlement(from_id)

    if not to_point:
        to_id = params.get('to_id')
        if not to_id and params.get('to_key'):
            to_id = params.get('to_key')[1:]
        to_point = get_settlement(to_id)

    return from_point, to_point


class NearDirectionsParams(FormBaseParams):
    from_id = fields.Int()
    to_id = fields.Int()

    from_key = SettlementKey()
    to_key = SettlementKey()

    from_point = ModelField(model=Settlement)
    to_point = ModelField(model=Settlement)

    distance = fields.Int(validate=validate.Range(min=50, max=350))

    @validates_schema
    def not_empty(self, data):
        if all(not data.get(i) for i in ['from_id', 'from_key', 'from_point']):
            raise ValidationError('from_id or from_key required', 'from_id')

        if all(not data.get(i) for i in ['to_id', 'to_key', 'to_point']):
            raise ValidationError('to_id or to_key required', 'to_id')


def get_settlements_within(to_settlement, from_settlement, distance, national_version):
    nearest = settlement_geo_index.get_nearest(to_settlement.id, distance)

    if settings.EXCLUDE_NEAR_POINTS_FROM_COUNTRIES:
        nearest = [s for s in nearest if str(s.country_id) not in settings.EXCLUDE_NEAR_POINTS_FROM_COUNTRIES]

    existing_arrivals = set(AviaDirectionNational.objects.filter(
        departure_settlement=from_settlement.id,
        arrival_settlement__in=[s.id for s in nearest],
        national_version=national_version
    ).values_list('arrival_settlement_id', flat=True))

    s_ids = [s.id for s in nearest if s.id in existing_arrivals]
    return list(Settlement.objects.filter(id__in=s_ids))


def get_directions_with_prices(city_from, settlements, params, passengers_key):
    return [DirectionPrice(
        city_from, settlement, None, flask.g.get('national_version'),
        date_forward=params.get('when'),
        date_backward=params.get('return_date'),
        passengers=passengers_key,
        klass=params.get('klass'),
        allow_roughly=True
    ) for settlement in settlements]


class NearDirectionsHandler(ApiHandler):
    PARAMS_SCHEMA = NearDirectionsParams
    TYPE_SCHEMA = OfferSchema
    MULTI = True

    def preprocess_fields(self, fields):
        if not fields:
            return [
                'from_city', 'to_city', 'direct_price', 'transfers_price',
                'date_forward', 'date_backward'
            ]

        return fields

    def _validate_points(self, from_point, to_point):
        return all(p and p.longitude and p.latitude for p in [from_point, to_point])

    def process(self, params, fields):
        from_settlement, to_settlement = get_settlements_from_params(params)

        if not self._validate_points(from_settlement, to_settlement):
            return None

        settlements = get_settlements_within(
            to_settlement, from_settlement, params.get('distance'),
            g.get('national_version')
        )
        passengers_key = get_passengers_key(params)

        return get_directions_with_prices(
            from_settlement, settlements, params, passengers_key
        )


class NearDistancesParams(Schema):
    from_id = fields.Int()
    to_id = fields.Int()

    from_key = SettlementKey()
    to_key = SettlementKey()

    from_point = ModelField(model=Settlement)
    to_point = ModelField(model=Settlement)


class NearDistanceSchema(TypeSchema):
    min_distance = fields.Int()
    max_distance = fields.Int()
    default_distance = fields.Int()


class NearDistancesHandler(ApiHandler):
    PARAMS_SCHEMA = NearDistancesParams
    TYPE_SCHEMA = NearDistanceSchema

    def process(self, params, fields):
        from_point, to_point = get_settlements_from_params(params)

        try:
            distances = NearDirection.objects.get(
                departure_settlement=from_point,
                arrival_settlement=to_point
            )
        except NearDirection.DoesNotExist:
            log.info('NearDirection.DoesNotExist %r %r', from_point, to_point)
            return None

        if not distances.default_distance:
            return None

        return distances


class NearestSettlementsParams(TypeSchema):
    point_id = fields.Int(required=True)
    count = fields.Int(required=False, missing=5)
    lang = fields.String(required=True)
    max_distance = fields.Int(required=False, missing=250)
    only_with_airports = fields.Bool(required=False, missing=True)


class NearestSettlementsSchema(Schema):
    settlement_id = fields.Int()
    title = fields.Str()
    airports = fields.List(fields.Int)


class NearestSettlementsHandler(ApiHandler):
    PARAMS_SCHEMA = NearestSettlementsParams
    TYPE_SCHEMA = NearestSettlementsSchema
    MULTI = True

    def _validate_point(self, point):
        return point and point.longitude and point.latitude

    def process(self, params, fields):
        point = get_settlement(params['point_id'])

        if not self._validate_point(point):
            return

        nearests = settlement_geo_index.get_nearest(point.id, params['max_distance'])

        found = 0
        max_count = params['count']
        for nearest in nearests:
            airports = geo_relations_repository.get_airport_ids_for(nearest.id)
            if not params['only_with_airports'] or airports:
                yield {
                    'settlement_id': nearest.id,
                    'title': nearest.get_title(params['lang']),
                    'airports': list(airports),
                }

                found += 1

            if found == max_count:
                break
