# -*- encoding: utf-8 -*-
from __future__ import absolute_import

from travel.avia.library.python.avia_data.models import NearCountries
from travel.avia.library.python.common.models.geo import Station
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api_types.country_directions import CountryDirectionsParams, get_settlement_and_country
from travel.avia.backend.main.api_types.country_offers import CountryOfferSchema
from travel.avia.backend.main.lib.directions import CountryDirectionFinder
from travel.avia.backend.main.lib.geo import get_model_query_set
from travel.avia.backend.main.lib.prices import multiply_price, AviaPrice
from .price import get_passengers_key


def get_near_countries(country):
    try:
        near = NearCountries.objects.get(country=country)

        result = []
        for c in near.neighbours.all():
            airports = Station.objects.filter(
                country=c,
                hidden=False,
                t_type_id=TransportType.PLANE_ID
            ).count()

            if airports:
                result.append(c)

        return result
    except NearCountries.DoesNotExist:
        return []


def get_country_prices(from_city, passengers_key, neighbours, when, return_date):
    neighbours_ids = [i.id for i in neighbours]

    qs = get_model_query_set(Station)

    arrival_ids = [s.settlement_id for s in qs.filter(
        hidden=False,
        t_type_id=TransportType.PLANE_ID,
        settlement__country_id__in=neighbours_ids
    ).only('settlement_id')]

    finder = CountryDirectionFinder(
        from_city, limit=10000, pad=0,
        when=when,
        return_date=return_date,
        arrival_ids=arrival_ids
    )

    min_prices = finder.find()
    prices_dict = {}

    # Нужна только мин цена, не важно прямая или пересадочная
    for min_price in min_prices:
        country_id = min_price.arrival_settlement.country_id
        price = AviaPrice(min_price.price, min_price.currency.code, min_price.currency.iso_code, roughly=True)
        price.value = multiply_price(price.value, passengers_key)
        if country_id not in prices_dict:
            prices_dict[country_id] = {
                'price': price,
                'date_forward': min_price.date_forward,
                'date_backward': min_price.date_backward,
            }
        elif prices_dict[country_id]['price'] > price:
            prices_dict[country_id] = {
                'price': price,
                'date_forward': min_price.date_forward,
                'date_backward': min_price.date_backward,
            }

    return prices_dict


def fill_neighbour_country(from_point, country_to_price, neighbour):
    price = country_to_price.get(neighbour.id)

    return {
        'city_from': from_point,
        'country_to': neighbour,
        'date_forward': price['date_forward'] if price else None,
        'date_backward': price['date_backward'] if price else None,
        'price': price['price'] if price else None,
    }


class NeighbourCountryDirectionsHandler(ApiHandler):
    PARAMS_SCHEMA = CountryDirectionsParams
    TYPE_SCHEMA = CountryOfferSchema
    MULTI = True

    def preprocess_fields(self, fields):
        if not fields:
            return [
                'from_city', 'to_country', 'price', 'date_forward', 'date_backward'
            ]

        return fields

    def process(self, params, fields):
        from_point, to_point = get_settlement_and_country(params)
        passengers_key = get_passengers_key(params)

        neighbours = get_near_countries(to_point)
        country_to_price = get_country_prices(
            from_city=from_point,
            passengers_key=passengers_key,
            neighbours=neighbours,
            when=params.get('when', None),
            return_date=params.get('return_date', None)
        )

        return [fill_neighbour_country(from_point, country_to_price, neighbour) for neighbour in neighbours]
