# -*- encoding: utf-8 -*-
from __future__ import absolute_import

import logging
import operator
from collections import namedtuple
from datetime import timedelta, date, datetime

from travel.avia.library.python.common.utils import environment

from django.conf import settings
from django.db import transaction
from django.db.models import Q, Sum
from django.db.models.aggregates import Max

from travel.avia.library.python.avia_data.models import (
    BalanceRedirectPrepeared,  BalancePriceListRedirectPrepared, ShowLog
)
from travel.avia.library.python.balance_client.client import Client as BalanceClient
from travel.avia.library.python.common.models.partner import Partner, UpdateHistoryRecord, PartnerUser

from travel.avia.backend.main.lib.blackbox_utils import get_uid
from travel.avia.backend.main.lib.partner_users import create_partner_user

log = logging.getLogger(__name__)

DEFAULT_PLATFORM_LIST = [502, 503, 504, 505, 508, 509]
NEW_SCHEME_START = date(2018, 3, 1)  # 1 марта перешли на новую схему. Раньше этой даты не посчитаны новые агрегаты


def enable_partner(params):
    Partner.objects.get(
        billing_order_id=params['campaign_id']
    ).enable(
        yandex_login=params['user']['yandex_login'],
        role=params['user']['role'],
    )


def disable_partner(params):
    Partner.objects.get(
        billing_order_id=params['campaign_id']
    ).disable(
        yandex_login=params['user']['yandex_login'],
        role=params['user']['role'],
    )


def get_last_history_records(action):
    return {
        status_update.partner_id: status_update
        for status_update in UpdateHistoryRecord.objects.filter(
            reduce(operator.or_, (
                Q(**latest_update_params)
                for latest_update_params in (
                    UpdateHistoryRecord.objects
                    .values('partner_id')
                    .filter(action=action)
                    .annotate(time=Max('time'))
                )), Q())
        )
    }


def partner_statuses(params):
    partners = list(
        Partner.objects.filter(
            Q(billing_order_id=params['campaign_id'])
            if 'campaign_id' in params
            else Q()
        )
    )

    history_update_active_status_by_partner = get_last_history_records(
        UpdateHistoryRecord.CHANGE_ACTIVE_STATUS_ACTION
    )
    history_update_unavailability_time_by_partner = get_last_history_records(
        UpdateHistoryRecord.CHANGE_UNAVAILABILITY_RANGE_ACTION
    )
    history_update_hidden_by_partner = get_last_history_records(
        UpdateHistoryRecord.CHANGE_HIDDEN_STATUS
    )

    result = []

    def serialize_history(history):
        if not history:
            return history

        return {
            'at': history.time.strftime('%Y-%m-%dT%H:%M:%S'),
            'by': {
                'yandex_login': history.updater_yandex_login,
                'role': history.updater_role,
            }
        }

    partners_data = [(
        partner,
        history_update_active_status_by_partner.get(partner.id),
        history_update_unavailability_time_by_partner.get(partner.id),
        history_update_hidden_by_partner.get(partner.id)
    ) for partner in partners]
    for partner, active_status, unavailability_status, hidden_status in partners_data:
        result.append({
            'id': partner.id,
            'code': partner.code,
            'disabled': partner.disabled,
            'title': partner.L_national_ru_title(),
            'campaign_id': partner.billing_order_id,
            'billing_client_id': partner.billing_client_id,
            # todo delete after 01.11.2016
            'updated': serialize_history(active_status),
            'active_status_history': serialize_history(active_status),
            'unabailable_status_history': serialize_history(unavailability_status),
            'hidden_status_history': serialize_history(hidden_status),
            'current_balance': '%.2f' % partner.current_balance,
            'click_price': '%.2f' % partner.click_price,
            'is_aviacompany': partner.is_aviacompany
        })

    return result


def get_user_data(user):
    return {
        'id': user.id,
        'role': user.role,
        'login': user.login,
        'name': user.name,
        'partner': {
            'campaignId': user.partner.billing_order_id
        } if user.partner else {}
    }


def get_partner_users(params):
    return [get_user_data(user) for user in list(PartnerUser.objects.all())]


def get_partner_user_by_login(params):
    try:
        return get_user_data(PartnerUser.objects.get(login=params['login']))
    except PartnerUser.DoesNotExist:
        return None


def update_unavailability_rule(params):
    user = params['user']
    billing_order_id = int(params['campaign_id'])

    def parse_date(raw_date):
        return datetime.strptime(raw_date, '%Y-%m-%dT%H:%M:%S.%fZ')

    start_unavailability_datetime = parse_date(params['start_unavailability_datetime'])
    end_unavailability_datetime = parse_date(params['end_unavailability_datetime'])

    Partner.objects.get(
        billing_order_id=billing_order_id
    ).update_unavailability_range(
        start_unavailability_datetime=start_unavailability_datetime,
        end_unavailability_datetime=end_unavailability_datetime,
        yandex_login=user['yandex_login'],
        role=user['role'],
    )


def generate_partner_code(title, repeat=0):
    code = filter(str.isalpha, str(title).lower())
    if repeat > 0:
        code += str(repeat)
    try:
        Partner.objects.get(code=code)
        return generate_partner_code(title, repeat=repeat+1)
    except Partner.DoesNotExist:
        return code


def create_partner(params):
    login = params['login']
    title = params['title']
    site_url = params['siteUrl']
    operator_uid = params['operatorUid']
    code = params.get('code')
    uid = get_uid(login)

    if not code:
        code = generate_partner_code(title)

    if not uid:
        raise Exception('unknown login: {}'.format(login))

    partners_by_bid = Partner.objects.exclude(billing_order_id__isnull=True).order_by('billing_order_id')
    bid = partners_by_bid.last().billing_order_id + 1 if len(partners_by_bid) else 1

    with transaction.atomic(using='writable'):
        partner = Partner(
            title=title,
            code=code,
            site_url=site_url,
            billing_order_id=bid
        )
        partner.save(using='writable')

        partner_user = PartnerUser(
            role='owner',
            login=login,
            name='',
            partner=partner,
            passportuid=uid,
        )
        partner_user.save(using='writable')

        client_id = BalanceClient(
            env=settings.YANDEX_ENVIRONMENT_TYPE,
            client_tvm_id=settings.TVM_SERVICE_ID,
            logger=log,
        ).create_avia_client(
            uid=uid,
            bid=bid,
            name=title,
            operator_uid=operator_uid
        )

        partner.billing_client_id = client_id
        partner.save(using='writable')


def update_partner_hidden_status(params):
    user = params['user']
    hidden = bool(params['hidden'])
    campaign_id = int(params['campaignId'])

    try:
        partner = Partner.objects.get(
            billing_order_id=campaign_id,
            hidden=not hidden
        )
    except Partner.DoesNotExist:
        return

    with transaction.atomic(using='writable'):
        UpdateHistoryRecord(
            partner=partner,
            updater_yandex_login=user['login'],
            updater_role=user['role'],
            action=UpdateHistoryRecord.CHANGE_HIDDEN_STATUS,
            description='hidden:[{}]'.format(hidden)
        ).save(using='writable')

        partner.hidden = hidden
        partner.save(using='writable')


def create_helper(params):
    operator_login = params['operatorLogin']
    login = params['login']
    operator_uid = params['operatorUid']
    role = params['role']
    uid = get_uid(login)

    if not uid:
        raise Exception('unknown login: {}'.format(login))

    if role not in ['helper', 'accessor']:
        raise Exception('The role is too high')

    operator_user = PartnerUser.objects.get(
        login=operator_login
    )

    if operator_user.role not in ['manager', 'owner', 'admin']:
        raise Exception('Operator role is too low')

    partner = Partner.objects.get(
        billing_order_id=params['campaign_id']
    )

    if not partner:
        raise Exception('Not valid campaign_id')

    balance = BalanceClient(env=settings.YANDEX_ENVIRONMENT_TYPE, client_tvm_id=settings.TVM_SERVICE_ID, logger=log)
    client = balance.get_passport_by_login(operator_uid, operator_login)

    with transaction.atomic(using='writable'):
        PartnerUser(
            role=role,
            login=login,
            name='',
            partner=partner
        ).save(using='writable')

        balance.create_avia_helper(
            uid=uid,
            operator_uid=operator_uid,
            client_id=client.id
        )


def delete_helper(params):
    operator_login = params['operatorLogin']
    operator_uid = params['operatorUid']
    login = params['login']
    uid = get_uid(login)

    partner = Partner.objects.get(
        billing_order_id=params['campaign_id']
    )

    if not partner:
        raise Exception('Not valid campaign_id')

    if not uid:
        raise Exception('unknown login: {}'.format(login))

    operator_user = PartnerUser.objects.get(
        login=operator_login
    )

    if operator_user.role not in ['manager', 'owner', 'admin']:
        raise Exception('Operator role is too low')

    helper_user = PartnerUser.objects.get(
        login=login,
        partner_id=operator_user.partner.id
    )

    if not helper_user:
        raise Exception('User with login {} is not found'.format(login))

    balance = BalanceClient(env=settings.YANDEX_ENVIRONMENT_TYPE, client_tvm_id=settings.TVM_SERVICE_ID, logger=log)
    client = balance.get_passport_by_login(operator_uid, operator_login)

    with transaction.atomic():
        helper_user.delete()

        balance.remove_avia_helper(
            uid=uid,
            operator_uid=operator_uid,
            client_id=client.id
        )


def get_last_week_clicks(billing_order_id):
    today = environment.today()
    week_ago = today - timedelta(weeks=1)
    last_week_filter = {
        'eventdate__gte': week_ago,
        'eventdate__lte': today,
    }

    if billing_order_id:
        last_week_filter['billing_order_id'] = billing_order_id

    return (
        BalanceRedirectPrepeared.objects
        .filter(**last_week_filter)
        .values('billing_order_id')
        .annotate(click_count=Sum('count'))
    )


def get_detailed_statistics(params):
    left_date = params['left_date']
    right_date = params['right_date']
    billing_order_id = params.get('billing_order_id')
    national_version_list = params.get('national_version_list')
    pp_list = params.get('pp_list')

    db_filter = {
        'eventdate__gte': left_date,
        'eventdate__lte': right_date
    }

    if billing_order_id:
        db_filter['billing_order_id'] = billing_order_id

    if national_version_list:
        db_filter['national_version__in'] = national_version_list.split(',')

    db_filter['pp__in'] = pp_list and pp_list.split(',') or DEFAULT_PLATFORM_LIST

    shows = (
        ShowLog.objects
        .values('billing_order_id', 'pp', 'national_version', 'eventdate', 'show_count')
        .filter(**db_filter)
    )

    clicks = (
        BalanceRedirectPrepeared.objects
        .values('billing_order_id', 'pp', 'national_version', 'eventdate')
        .annotate(click_count=Sum('count'))
        .filter(**db_filter)
    )

    last_week_clicks = get_last_week_clicks(billing_order_id)

    return {
        'shows': [{
            'billing_order_id': s['billing_order_id'],
            'pp': s['pp'],
            'national_version': s['national_version'],
            'eventdate': s['eventdate'].isoformat(),
            'show_count': s['show_count']
        } for s in shows],

        'clicks': [{
            'billing_order_id': c['billing_order_id'],
            'pp': c['pp'],
            'national_version': c['national_version'],
            'eventdate': c['eventdate'].isoformat(),
            'click_count': c['click_count']
        } for c in clicks],

        'last_week_clicks_by_partner': {c['billing_order_id']: c['click_count'] for c in last_week_clicks}
    }


def get_partner_login(params):
    partner = Partner.objects.filter(billing_order_id=params['billing_order_id'])
    return PartnerUser.objects.values('login').get(partner=partner, role='owner')['login']


def get_detailed_statistics_new_scheme(params):
    Key = namedtuple('Key', ('billing_order_id', 'pp', 'national_version', 'eventdate'))
    left_date = datetime.strptime(params['left_date'], "%Y-%m-%d").date()
    right_date = datetime.strptime(params['right_date'], "%Y-%m-%d").date()
    billing_ids_map = dict(
        Partner.objects.all()
        .values_list('billing_client_id', 'billing_order_id')
    )

    db_filter = {}

    pp_list = params.get('pp_list')
    db_filter['pp__in'] = pp_list and pp_list.split(',') or DEFAULT_PLATFORM_LIST

    national_version_list = params.get('national_version_list')
    if national_version_list:
        db_filter['national_version__in'] = national_version_list.split(',')

    new_scheme_db_filter = db_filter.copy()

    db_filter.update({
        'eventdate__gte': left_date,
        'eventdate__lte': right_date
    })

    new_scheme_db_filter.update({
        'event_date__gte': max(NEW_SCHEME_START, left_date),  # можно и не ограничивать слева, там все равно не должно быть данных
        'event_date__lte': right_date
    })

    billing_order_id = params.get('billing_order_id')
    if billing_order_id:
        db_filter['billing_order_id'] = billing_order_id

        p = Partner.objects.get(billing_order_id=billing_order_id)
        billing_client_id = p.billing_client_id
        new_scheme_db_filter['billing_client_id'] = billing_client_id

    show_keys = Key._fields + ('show_count',)
    shows = ShowLog.objects.values_list(*show_keys).filter(**db_filter)

    shows_by_key = {Key(*s[:4]): s[4] for s in shows}

    click_keys = Key._fields + ('count',)
    clicks = (
        BalanceRedirectPrepeared.objects
        .values_list(*click_keys)
        .filter(**db_filter)
    )

    old_scheme_clicks_by_key = {Key(*c[:4]): c[4] for c in clicks}

    clicks_with_costs = (
        BalancePriceListRedirectPrepared.objects
        .values('billing_client_id', 'pp', 'national_version', 'event_date', 'count', 'cost', 'payments_count')
        .filter(**new_scheme_db_filter)
    )
    result_clicks_with_costs = {}

    for c in clicks_with_costs:
        if c['billing_client_id'] not in billing_ids_map:
            log.warning('Unknown billing_client_id %s')
            continue
        billing_order_id = billing_ids_map[c['billing_client_id']]
        key = Key(billing_order_id, c['pp'], c['national_version'], c['event_date'])
        result_clicks_with_costs[key] = dict(
            key._asdict(),
            count=c['count'],
            show_count=shows_by_key.pop(key, 0),
            cost=c['cost'],
            payments_count=c['payments_count'],
        )

    for key, count in old_scheme_clicks_by_key.iteritems():
        if key not in result_clicks_with_costs:
            result_clicks_with_costs[key] = dict(
                key._asdict(),
                count=count,
                show_count=shows_by_key.pop(key, 0),
                cost=0.,
                payments_count=0.,
            )

    # Дописываем оставшиеся показы дней, когда небыло кликов
    for key, show_count in shows_by_key.iteritems():
        result_clicks_with_costs[key] = dict(
            key._asdict(),
            show_count=show_count,
            count=0.,
            cost=0.,
            payments_count=0.,
        )

    return [{
        'billing_order_id': c['billing_order_id'],
        'pp': c['pp'],
        'national_version': c['national_version'],
        'eventdate': c['eventdate'].isoformat(),
        'show_count': c['show_count'],
        'click_count': c['count'],
        'sum_cost': c['cost'],
        'payments_count': c['payments_count'],
        'conversion': c['payments_count'] / float(c['count']) if c['count'] else 0.,
    } for c in result_clicks_with_costs.itervalues()]


def create_partner_user_handler(params):
    create_partner_user(params['role'], params['login'], params['name'], params.get('partnerCode'))
    return 'ok'
