# -*- encoding: utf-8 -*-
from __future__ import absolute_import

from flask import g
from marshmallow import Schema, fields

from travel.avia.library.python.avia_data.models import AmadeusMerchant
from travel.avia.library.python.common.models.partner import DohopVendor, Partner

from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.api_handler import ApiHandler


def get_partners_by_codes(codes):
    partner_codes = []
    dohop_ids = []
    amadeus_codes = []
    partners = []

    for c in codes:
        if c.startswith('dohop_'):
            try:
                dohop_ids.append(int(c.replace('dohop_', '')))

            except ValueError:
                continue
        elif c.startswith('amadeus_'):
            amadeus_codes.append(c)
        else:
            partner_codes.append(c)

    if partner_codes:
        partners += [p for p in Partner.objects.filter(code__in=partner_codes)]

    if dohop_ids:
        partners += [v for v in DohopVendor.objects.filter(dohop_id__in=dohop_ids)]

    if amadeus_codes:
        partners += [v for v in AmadeusMerchant.objects.filter(code__in=amadeus_codes)]

    return partners


def get_partner_by_code(code):
    partners = get_partners_by_codes([code])
    try:
        return partners[0]
    except IndexError:
        return None


class PartnersParams(Schema):
    include_dohop = fields.Bool(missing=True)
    mobile = fields.Bool(missing=False)


class PartnerParams(Schema):
    code = fields.Str(required=True)


class StaticPageSchema(TypeSchema):
    title = fields.Str(attribute='L_title')
    url = fields.Str()
    content = fields.Str(attribute='L_content')
    logo = fields.Function(lambda p: p.logo.url if p.logo else None)


class PartnerSchema(TypeSchema):
    id = fields.Int()
    title = fields.Method('get_title')
    logo = fields.Function(lambda p: p.get_national_logo(g.get('national_version')))
    logo_svg = fields.Function(lambda p: p.get_national_logo_svg(g.get('national_version')))
    icon_svg = fields.Function(lambda p: p.get_icon_svg())

    def get_title(self, p):
        field = 'L_national_%s_title' % g.get('national_version')
        L_national_title = getattr(p, field, p.L_title)
        return L_national_title()


class DummyPartnerSchema(TypeSchema):
    id = fields.Int()
    title = fields.Str()
    logo = fields.Str()
    logo_svg = fields.Str()


class PartnerHandler(ApiHandler):
    PARAMS_SCHEMA = PartnerParams
    TYPE_SCHEMA = PartnerSchema

    def process(self, params, fields):
        return get_partner_by_code(params.get('code'))
