# -*- encoding: utf-8 -*-
from __future__ import absolute_import

from collections import defaultdict

from cachetools.func import ttl_cache
from django.conf import settings
from marshmallow import Schema, fields, pre_dump

from travel.avia.library.python.common.models.partner import Partner, CPCPrice

from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.api_schema import TypeSchema


class _CPCSource:
    def __init__(self, name):
        self.name = name

    def get_field_name(self):
        if not self.name:
            return 'eCPC'
        return 'eCPC_%s' % self.name

    def get_cpc_value(self, cpc_price):
        return getattr(cpc_price, self.get_field_name())

    def get_api_field_name(self, national):
        if not self.name:
            return national
        return '%s_%s' % (self.name, national)


_sources = [
    _CPCSource(''),
    _CPCSource('rasp_direct'),
    _CPCSource('sovetnik_direct'),
    _CPCSource('wizard_direct'),
    _CPCSource('wizard_indirect'),
]


@ttl_cache(maxsize=None, ttl=5 * 60)
def _click_prices_by_partner_id():
    cpc_prices = CPCPrice.objects.select_related()
    prices_by_partner_id = defaultdict(dict)
    for c in cpc_prices:
        for source in _sources:
            cpc_value = source.get_cpc_value(c)
            api_field_name = source.get_api_field_name(c.national_version.code)
            prices_by_partner_id[c.partner.id][api_field_name] = cpc_value
    return prices_by_partner_id


class PartnersDataSchema(TypeSchema):
    id = fields.Int()
    code = fields.Str()
    click_price = fields.Float()
    billing_client_id = fields.Int()
    billing_order_id = fields.Int()
    billing_datasource_id = fields.Int()
    enabled = fields.Boolean()
    click_price_national = fields.Dict(fields.String())
    e_click_price_national = fields.Dict(fields.String())
    marker = fields.Str()
    enabled_in_wizard = fields.Dict(fields.Boolean())
    title = fields.Str()
    site_url = fields.Str()
    can_sale_by_installments = fields.Boolean()

    @pre_dump
    def get_price_national(self, partner):
        partner.click_price_national = {
            national: str(partner.get_national_click_price(national))
            for national in settings.AVIA_NATIONAL_VERSIONS
        }

    @pre_dump
    def get_e_click_price_national(self, partner):
        cpc_price_by_national = _click_prices_by_partner_id()[partner.id]
        partner.e_click_price_national = {}

        for national in settings.AVIA_NATIONAL_VERSIONS:
            for source in _sources:
                api_field_name = source.get_api_field_name(national)
                cpc_value = str(cpc_price_by_national.get(api_field_name, 0))
                partner.e_click_price_national[api_field_name] = cpc_value

    @pre_dump
    def get_enabled_in_wizard(self, partner):
        partner.enabled_in_wizard = {
            national: getattr(partner, 'enabled_in_wizard_{}'.format(national))
            for national in settings.AVIA_WIZARD_NATIONAL_VERSIONS
        }


class PartnersDataHandler(ApiHandler):
    PARAMS_SCHEMA = Schema
    TYPE_SCHEMA = PartnersDataSchema
    MULTI = True

    def process(self, params, fields):
        return Partner.objects.all()
