# -*- encoding: utf-8 -*-
from __future__ import absolute_import

from datetime import timedelta, date

import flask
from django.conf import settings
from marshmallow import Schema, fields, validate

from travel.avia.library.python.avia_data.models import MinPrice
from travel.avia.library.python.common.models.currency import Currency
from travel.avia.library.python.common.models.geo import Point, Settlement

from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.lib.geo import get_settlement_by_point
from travel.avia.backend.main.lib.prices import multiply_price, key_by_params, get_min_prices_by_keys, AviaPrice
from travel.avia.backend.repository.currency import currency_repository


class PriceSchema(TypeSchema):
    value = fields.Int()
    base_value = fields.Int()
    currency = fields.Str()
    roughly = fields.Bool()


# Не уверен что это должно лежать здесь, но пока пусть лежит
class MinPricesSchema(TypeSchema):
    direct = fields.Nested(PriceSchema)
    transfers = fields.Nested(PriceSchema)


class DatePriceSchema(TypeSchema):
    date = fields.Date()
    prices = fields.Nested(MinPricesSchema)


def get_passengers_key(params):
    return '_'.join([
        str(params.get(i)) for i in ['adult_seats', 'children_seats', 'infant_seats']
    ])


def _get_cache_key(from_point, to_point, date_forward, delta, direct,
                   passengers_key, force_one_passenger):
    if delta is not None:
        date_backward = date_forward + timedelta(days=delta)
    else:
        date_backward = None

    return key_by_params(city_from_key=from_point.point_key,
                         city_to_key=to_point.point_key,
                         national_version=flask.g.get('national_version'),
                         passengers_key='1_0_0' if force_one_passenger else passengers_key,
                         date_forward=date_forward,
                         date_backward=date_backward,
                         is_direct=direct)


def get_prices_from_db(from_settlement, to_settlement, left_date, right_date, delta,
                       passengers_key, force_one_passenger, airline_id=None, direct=None):
    min_prices = MinPrice.objects.filter(
        national_version=flask.g.get('national_version'),
        departure_settlement=from_settlement,
        arrival_settlement=to_settlement,
        date_forward__range=(left_date, right_date),
        passengers='1_0_0' if force_one_passenger else passengers_key
    ).order_by('date_forward')

    if delta is None:
        min_prices = min_prices.filter(date_backward__isnull=True)
    else:
        min_prices = min_prices.extra(where=[
            'date_backward = DATE_ADD(date_forward, INTERVAL %s DAY)'
        ], params=[delta])

    if airline_id:
        min_prices = min_prices.filter(airlines__id=airline_id)

    if direct:
        min_prices = min_prices.filter(direct_flight=True)

    days_price = {}

    min_prices = list(min_prices)

    currency_ids = set([p.currency_id for p in min_prices])
    currencies = {c.id: c for c in list(Currency.objects.filter(id__in=currency_ids))}

    for min_price in min_prices:
        date_forward = min_price.date_forward
        prices = days_price.setdefault(date_forward, {})
        min_price.currency = currencies.get(min_price.currency_id, None)

        key = 'direct' if min_price.direct_flight else 'transfers'
        prices[key] = AviaPrice(
            multiply_price(min_price.price, passengers_key),
            min_price.currency.code,
            min_price.currency.iso_code,
            roughly=True
        )

    return days_price


def safe_get_minprice(cache, key, airline_id):
    data = cache.get(key)

    if not airline_id or not data:
        return data

    # todo kill after 2017.06.01
    if 'companies' not in data:
        return None

    if airline_id not in data['companies']:
        return None

    return data


def get_prices_from_cache(from_settlement, to_settlement, left_date, right_date, delta,
                          passengers_key, force_one_passenger, airline_id=None, direct=None):
    currency = settings.AVIA_NATIONAL_CURRENCIES.get(
        flask.g.get('national_version'),
        settings.AVIA_NATIONAL_CURRENCIES['ru']
    )
    iso_currency = currency_repository.get_by_code(currency).iso_code

    cache_keys = []

    day = left_date
    while day <= right_date:
        cache_keys.append(_get_cache_key(
            from_settlement, to_settlement, day, delta,
            True, passengers_key, force_one_passenger
        ))

        if direct is None:
            cache_keys.append(_get_cache_key(
                from_settlement, to_settlement, day, delta,
                False, passengers_key, force_one_passenger
            ))

        day += timedelta(days=1)

    cached_prices = get_min_prices_by_keys(cache_keys)

    days_price = {}

    day = left_date
    while day <= right_date:
        direct_cache_key = _get_cache_key(
            from_settlement, to_settlement, day, delta,
            True, passengers_key, force_one_passenger
        )
        direct_cached_price = safe_get_minprice(cached_prices, direct_cache_key, airline_id)

        indirect_cached_price = None
        if direct is None:
            indirect_cache_key = _get_cache_key(
                from_settlement, to_settlement, day, delta,
                False, passengers_key, force_one_passenger
            )
            indirect_cached_price = safe_get_minprice(cached_prices, indirect_cache_key, airline_id)

        if direct_cached_price or indirect_cached_price:
            prices = days_price.setdefault(day, {})

        if direct_cached_price:
            price = direct_cached_price['price']
            if force_one_passenger:
                price = multiply_price(direct_cached_price['price'], passengers_key)

            prices['direct'] = AviaPrice(price, currency, iso_currency, roughly=force_one_passenger)

        if indirect_cached_price:
            price = indirect_cached_price['price']
            if force_one_passenger:
                price = multiply_price(indirect_cached_price['price'], passengers_key)

            prices['transfers'] = AviaPrice(price, currency, iso_currency, roughly=force_one_passenger)

        day += timedelta(days=1)

    return days_price


class MinPricesParams(Schema):
    from_point_key = fields.Str(required=True)
    to_point_key = fields.Str(required=True)

    # даты внутри которых нужно производить поиск
    left_date = fields.Date(required=True)
    right_date = fields.Date(required=True)

    # для двустороннего поиска
    delta = fields.Integer(validate=validate.Range(min=0, max=366))

    adult_seats = fields.Integer(required=True)
    children_seats = fields.Integer(required=True)
    infant_seats = fields.Integer(required=True)

    airline_id = fields.Int()
    direct = fields.Bool()


class MinPricesHandler(ApiHandler):
    PARAMS_SCHEMA = MinPricesParams
    TYPE_SCHEMA = DatePriceSchema
    MULTI = True

    def process(self, params, fields):
        from_key = params.get('from_point_key')
        to_key = params.get('to_point_key')

        try:
            from_settlement = get_settlement_by_point(Point.get_by_key(from_key))
            to_settlement = get_settlement_by_point(Point.get_by_key(to_key))
        except Settlement.DoesNotExist:
            return None

        if not from_settlement or not to_settlement:
            return None

        now = flask.request.environ['now']
        left_date = params.get('left_date')
        # игнорируем данные за прошлые месяца
        if left_date.year < now.year or (left_date.year == now.year and left_date.month < now.month):
            left_date = date(now.year, now.month, 1)
        right_date = params.get('right_date')
        delta = params.get('delta')

        passengers_key = get_passengers_key(params)
        many_passegers = passengers_key != '1_0_0'

        airline_id = params.get('airline_id')
        direct = params.get('direct')

        # Всегда ищем для одного пассажира, т.к. для многих - это редкость
        days_db_prices = get_prices_from_db(
            from_settlement, to_settlement,
            left_date, right_date, delta,
            passengers_key, force_one_passenger=many_passegers,
            airline_id=airline_id, direct=direct
        )

        days_cached_prices = get_prices_from_cache(
            from_settlement, to_settlement,
            left_date, right_date, delta,
            passengers_key, force_one_passenger=many_passegers,
            airline_id=airline_id, direct=direct
        )

        days_price = days_db_prices.copy()
        days_price.update(days_cached_prices)

        # # Ну и на всякий случай обновим настоящим данными
        if many_passegers:
            days_db_prices_real = get_prices_from_db(
                from_settlement, to_settlement,
                left_date, right_date, delta,
                passengers_key, force_one_passenger=False,
                airline_id=airline_id, direct=direct
            )
            days_cached_prices_real = get_prices_from_cache(
                from_settlement, to_settlement,
                left_date, right_date, delta,
                passengers_key, force_one_passenger=False,
                airline_id=airline_id, direct=direct
            )

            days_price.update(days_db_prices_real)
            days_price.update(days_cached_prices_real)

        data = []

        day = left_date
        while day <= right_date:
            prices = days_price.get(day, {})
            # в текущем месяце игнорируем цены в прошлом
            if now.year == day.year and now.month == day.month and now.day > day.day:
                prices = None

            data.append({
                'date': day,
                'prices': prices or None
            })

            day += timedelta(days=1)

        return data
