# -*- encoding: utf-8 -*-
from __future__ import absolute_import

import flask

from django.utils.translation import get_language
from marshmallow import fields

from travel.avia.library.python.avia_data.models import AviaRecipe

from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.fields import Related
from travel.avia.backend.main.lib.recipe_provider import recipe_provider
from travel.avia.backend.main.lib.recipes.popular import popular_recipe
from .recipe_offers import RecipeOffersHandler
from .airline_recipe import AirlineRecipeOffersHandler
from .company import AirlineHandler
from .reference import FormBaseParams
from .settlement import get_settlement


def get_recipes(settlement, when, return_date, national_version):
    if not settlement:
        return None

    recipes = recipe_provider.get_actual(
        settlement=settlement,
        national_version=national_version
    )

    if when:
        recipes = [
            r for r in recipes
            if r.date_start is None or r.date_start <= when
        ]
    if return_date:
        recipes = [
            r for r in recipes
            if r.date_end is None or r.date_end >= return_date
        ]

    def one_and_client_filter(recipe):
        regions_ids = recipe.region_ids
        if len(regions_ids) == 1 and list(regions_ids)[0] == settlement.region_id:
            return False

        settlement_ids = recipe.settlement_ids
        if len(settlement_ids) == 1 and list(settlement_ids)[0] == settlement.pk:
            return False

        return True

    recipes = filter(one_and_client_filter, recipes)

    return recipes


class RecipesParams(FormBaseParams):
    from_id = fields.Field(required=True)
    oneway = fields.Boolean(missing=False)
    slug = fields.String()


class RecipeParams(RecipesParams):
    id = fields.Int(required=True)


def copy_param(name):
    def wrapper(obj, context):
        return context['params'].get(name)

    return wrapper


def copy_date_param(name):
    def wrapper(obj, context):
        val = context['params'].get(name)
        if val:
            return val.isoformat()

        return None

    return wrapper


class RecipeSchema(TypeSchema):
    id = fields.Int()
    title = fields.Str(attribute='L_title')
    h1 = fields.Str(attribute='L_h1')
    meta_title = fields.Str(attribute='L_meta_title')
    meta_description = fields.Str(attribute='L_meta_description')
    meta_title_key = fields.Str()

    recipe_offers = Related(handler=RecipeOffersHandler, params={
        'recipe': 'self',
        'from_id': copy_param('from_id'),
        'when': copy_date_param('when'),
        'return_date': copy_date_param('return_date'),
        'oneway': copy_param('oneway'),
        'adult_seats': copy_param('adult_seats'),
        'children_seats': copy_param('children_seats'),
        'infant_seats': copy_param('infant_seats'),
        'klass': copy_param('klass'),
        'limit': copy_param('limit'),
    })

    airline_recipe_offers = Related(handler=AirlineRecipeOffersHandler, params={
        'recipe': 'self',
        'from_id': copy_param('from_id'),
        'when': copy_date_param('when'),
        'return_date': copy_date_param('return_date'),
        'oneway': copy_param('oneway'),
        'adult_seats': copy_param('adult_seats'),
        'children_seats': copy_param('children_seats'),
        'infant_seats': copy_param('infant_seats'),
        'klass': copy_param('klass'),
        'limit': copy_param('limit'),
        'slug': copy_param('slug'),
    })

    airline = Related(handler=AirlineHandler, params={
        'slug': copy_param('slug'),
    })


class RecipeHandler(ApiHandler):
    PARAMS_SCHEMA = RecipeParams
    TYPE_SCHEMA = RecipeSchema

    def preprocess_fields(self, fields):
        if not fields:
            return ['id', 'title']

        return fields

    def process(self, params, fields):
        if params.get('id') == 0:
            return popular_recipe

        return AviaRecipe.objects.get(id=params.get('id'))


class RecipesHandler(ApiHandler):
    PARAMS_SCHEMA = RecipesParams
    TYPE_SCHEMA = RecipeSchema
    MULTI = True
    IS_RAW_SCHEMA = True

    def preprocess_fields(self, fields):
        if not fields:
            return ['id', 'title']

        return fields

    def process(self, params, fields):
        city_from = get_settlement(params.get('from_id'))
        if not city_from:
            return None

        recipes = get_recipes(
            settlement=city_from,
            when=fields.get('when'),
            return_date=fields.get('return_date'),
            national_version=flask.g.national_version
        )

        lang = get_language()
        recipe_id = params.get('recipe_id')

        return [
            self.serialize_recipe(
                r,
                lang,
                fields,
                params,
                r.pk == recipe_id
            )
            for r in recipes
        ]

    def serialize_recipe(self, r, lang, fields, params, is_current):
        data = {
            "h1": r.get_h1(lang),
            "id": r.pk,
            "metaDescription": r.get_meta_description(lang),
            "metaTitle": r.get_meta_title(lang),
            "metaTitleKey": r.meta_title_key,
            "recipeOffers": None,
            "airlineRecipeOffers": None,
            "title": r.get_title(lang)
        }

        if 'recipe_offers' in fields:
            recipe_offers_params = fields['recipe_offers']['params']
            if recipe_offers_params.get('recipeId') == r.pk:
                data["recipeOffers"] = RecipeOffersHandler().process(
                    {
                        'recipe_id': r.pk,
                        'from_id': params.get('from_id'),
                        'when': params.get('when'),
                        'return_date': params.get('return_date'),
                        'oneway': params.get('oneway'),
                        'adult_seats': params.get('adult_seats'),
                        'children_seats': params.get('children_seats'),
                        'infant_seats': params.get('infant_seats'),
                        'klass': params.get('klass'),
                        'limit': recipe_offers_params.get('limit'),
                    }, {}
                )
        if 'airline_recipe_offers' in fields and is_current:
            airline_recipe_offers_params = fields['airline_recipe_offers']['params']
            if airline_recipe_offers_params.get('recipeId') == r.pk:
                data["airlineRecipeOffers"] = AirlineRecipeOffersHandler().process(
                    {
                        'recipe_id': r.pk,
                        'from_id': params.get('from_id'),
                        'when': params.get('when'),
                        'return_date': params.get('return_date'),
                        'oneway': params.get('oneway'),
                        'adult_seats': params.get('adult_seats'),
                        'children_seats': params.get('children_seats'),
                        'infant_seats': params.get('infant_seats'),
                        'klass': params.get('klass'),
                        'limit': params.get('limit'),
                        'slug': params.get('slug')
                    }, {}
                )

        return data
