# -*- encoding: utf-8 -*-
from __future__ import absolute_import

import flask

from django.utils.translation import get_language
from marshmallow import fields

from travel.avia.library.python.avia_data.models import AviaRecipe
from travel.avia.library.python.common.models.geo import Settlement

from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.fields import ModelField
from travel.avia.backend.main.lib.directions import DirectionRawSQLFinder, apply_passengers
from travel.avia.backend.main.lib.geo import has_airport
from travel.avia.backend.main.lib.recipes import get_recipe_offers
from travel.avia.backend.main.lib.recipes.popular import get_popular_directions, popular_recipe
from travel.avia.backend.main.lib.settlement_serializer import SettlementSerializer
from .offers import OfferSchema
from .reference import FormBaseParams
from .settlement import (get_settlement, get_country,
                         fill_related_fields as fill_settlement_related_fields)
from .settlement_image import get_settlement_images

DEFAULT_LIMIT = 36


def get_extra_ids(extra_geo_ids):
    extra_ids = []
    for geo_id in extra_geo_ids:
        # из user buckets может приходить много geoid которых мы не знаем
        # чтобы не делать запросы к бд идём напрямую в кэш
        if Settlement.objects.precached:
            s = Settlement.objects.precache_caches['_geo_id_exact'].get(geo_id)
            if s:
                extra_ids.append(s[0].id)
        else:
            try:
                extra_ids.append(Settlement.objects.get(_geo_id=geo_id).id)
            except Settlement.DoesNotExist:
                pass
    return extra_ids


def get_offers(recipe, city_from, when, return_date, passengers, extra_geo_ids, national_version, limit):
    limit = limit or DEFAULT_LIMIT

    if recipe.id == 0:
        extra_ids = get_extra_ids(extra_geo_ids)

        return get_popular_directions(
            city_from, when, return_date, passengers, extra_ids, limit
        )

    return get_recipe_offers(recipe, city_from, when, return_date, passengers, national_version, limit)


def get_offer(city_from, city_to, when, return_date, passengers):
    d = DirectionRawSQLFinder(city_from, when, return_date).get_direction(city_to, city_from)

    return apply_passengers(passengers, [d])[0]


class BaseOfferParams(FormBaseParams):
    from_id = fields.Int(required=True)
    oneway = fields.Boolean(missing=False)


class RecipeOffersParams(BaseOfferParams):
    recipe_id = fields.Integer(required=True)
    extra_ids = fields.List(fields.Int)
    limit = fields.Integer(allow_none=True)

    recipe = ModelField(model=AviaRecipe)


def fill_image_in_offers(offers):
    images = get_settlement_images([offer.city_to.id for offer in offers])
    for i in xrange(0, len(images)):
        offers[i].image = images[i]


class RecipeOffersHandler(ApiHandler):
    """ Получение списка предложений в рецепте """

    PARAMS_SCHEMA = RecipeOffersParams
    TYPE_SCHEMA = OfferSchema
    MULTI = True
    IS_RAW_SCHEMA = True

    def preprocess_fields(self, fields):
        if not fields:
            return [
                'from_city', 'to_city', 'direct_price', 'transfers_price',
                'date_forward', 'date_backward'
            ]

        return fields

    def _make_passengers(self, params):
        return {
            'adult_seats': params.get('adult_seats', None) or 1,
            'children_seats': params.get('children_seats', None) or 0,
            'infant_seats': params.get('infant_seats', None) or 0,
        }

    def _make_recipe(self, params):
        recipe = params.get('recipe')

        if recipe:
            if recipe.id != params['recipe_id']:
                return None

        else:
            recipe_id = params.get('recipe_id')

            if recipe_id == 0:
                recipe = popular_recipe
            else:
                recipe = AviaRecipe.objects.get(id=recipe_id)

        return recipe

    def process(self, params, fields):
        national_version = flask.g.get('national_version')
        recipe = self._make_recipe(params)

        if not recipe:
            return None

        city_from = get_settlement(params.get('from_id'))

        offers = get_offers(
            recipe, city_from,
            params.get('when'), params.get('return_date'),
            self._make_passengers(params),
            params.get('extra_ids', None) or [],
            national_version,
            params.get('limit', None)
        )

        fill_settlement_related_fields([o.city_to for o in offers])
        fill_image_in_offers(offers)

        settlement_serializer = SettlementSerializer()
        lang = get_language()
        nv = flask.g.national_version
        return [
            self._serialize_offer(o, settlement_serializer, nv, lang) for o in offers
        ]

    def _serialize_offer(self, offer, settlement_serializer, nv, lang):

        return {
            'fromCity': settlement_serializer.serialize(
                offer.city_from,
                nv,
                lang
            ),
            'toCity': settlement_serializer.serialize(
                offer.city_to,
                nv,
                lang
            ),
            'directPrice': {
                'value': offer.direct_price.value,
                'baseValue': offer.direct_price.base_value,
                'currency': offer.direct_price.currency,
                'roughly': offer.direct_price.roughly,
            } if offer.direct_price else None,
            'transfersPrice': {
                'value': offer.indirect_price.value,
                'base_value': offer.indirect_price.base_value,
                'currency': offer.indirect_price.currency,
                'roughly': offer.indirect_price.roughly,
            } if offer.indirect_price else None,
            'price': {
                'value': offer.price.value,
                'baseValue': offer.price.base_value,
                'currency': offer.price.currency,
                'roughly': offer.price.roughly,
            } if offer.price else None,
            'image': {
                'url2': offer.image.url2,
                'id': offer.image.pk
            } if offer.image else None,
            'dateForward': offer.date_forward and offer.date_forward.strftime('%Y-%m-%d'),
            'dateBackward': offer.date_backward and offer.date_backward.strftime('%Y-%m-%d'),
        }


class RecipeOfferParams(BaseOfferParams):
    to_id = fields.Int(required=True)


class RecipeOfferHandler(RecipeOffersHandler):
    """ Получение предложения по параметрам """

    PARAMS_SCHEMA = RecipeOfferParams
    TYPE_SCHEMA = OfferSchema
    MULTI = False

    def process(self, params, fields):
        city_from = get_settlement(params.get('from_id'))
        city_to = get_settlement(params.get('to_id'))

        if not city_from or not city_to or not has_airport(city_to):
            return None

        offer = get_offer(
            city_from, city_to,
            params.get('when'), params.get('return_date'),
            self._make_passengers(params),
        )

        offer.city_to.country = get_country(offer.city_to)
        fill_image_in_offers([offer])

        settlement_serializer = SettlementSerializer()
        lang = get_language()
        nv = flask.g.national_version
        return self._serialize_offer(
            offer,
            settlement_serializer,
            nv,
            lang
        )
