# -*- encoding: utf-8 -*-
from __future__ import absolute_import

from marshmallow import Schema, fields

from travel.avia.library.python.avia_data.models.review import FlightReview, FlightNumber

from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.api_handler import ApiHandler


class ReviewByFlightParams(Schema):
    flight_numbers = fields.List(fields.Str())


class ReviewByAirlineParams(Schema):
    limit = fields.Int()
    airline_id = fields.Int()
    exclude_flight_numbers = fields.List(fields.Str())


class ReviewSchema(TypeSchema):
    id = fields.Int()
    review_content = fields.Str()
    author_name = fields.Str()
    review_datetime = fields.DateTime()
    flight_numbers = fields.List(fields.Str())
    review_url = fields.Str()
    source = fields.Str()


class ReviewListSchema(TypeSchema):
    review_list = fields.List(fields.Nested(ReviewSchema))


def reviews_to_json(reviews):
    review_list = []
    for review in reviews:
        review_list.append({
            'id': review.id,
            'flight_numbers': [fn.flight_number for fn in review.flight_numbers.all()],
            'review_content': review.review_content,
            'author_name': review.author_name,
            'review_datetime': review.review_datetime,
            'review_url': review.review_url,
            'source': review.source,
        })

    return {
        'review_list': review_list
    }


class ReviewListHandler(ApiHandler):
    PARAMS_SCHEMA = ReviewByFlightParams
    TYPE_SCHEMA = ReviewListSchema

    def process(self, params, fields):
        flight_numbers = params.get('flight_numbers')

        flight_number_models = FlightNumber.objects.filter(flight_number__in=flight_numbers)

        review_models = FlightReview.objects.filter(flight_numbers__in=flight_number_models, enable_show=True) \
            .distinct().order_by('-review_datetime').prefetch_related('flight_numbers', 'source')

        return reviews_to_json(review_models)


class AirlineReviewsHandler(ApiHandler):
    PARAMS_SCHEMA = ReviewByAirlineParams
    TYPE_SCHEMA = ReviewListSchema

    def process(self, params, fields):
        limit = params.get('limit')
        airline_id = params.get('airline_id')
        flight_numbers = params.get('exclude_flight_numbers', [])

        flight_number_models = FlightNumber.objects.filter(flight_number__in=flight_numbers)

        review_models = FlightReview.objects \
            .filter(airline_id=airline_id, enable_show=True) \
            .exclude(flight_numbers__in=flight_number_models) \
            .distinct().order_by('-review_datetime').prefetch_related('flight_numbers', 'source')

        if limit is not None:
            review_models = review_models[:limit]

        return reviews_to_json(review_models)
