# -*- encoding: utf-8 -*-
from __future__ import absolute_import

from datetime import datetime

from marshmallow import Schema, fields

from travel.avia.library.python.common.models.geo import Station
from travel.avia.library.python.route_search.shortcuts import find as route_find

from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.fields import PointKey
from travel.avia.backend.main.lib.geo import get_settlement_by_point


class ScheduleParams(Schema):
    from_point_key = PointKey(required=True)
    to_point_key = PointKey(required=True)

    left_date = fields.Date(required=True)
    right_date = fields.Date(required=True)


class ScheduleSchema(TypeSchema):
    has_routes = fields.Bool()


class ScheduleHandler(ApiHandler):
    PARAMS_SCHEMA = ScheduleParams
    TYPE_SCHEMA = ScheduleSchema

    def process(self, params, fields):
        from_model, from_id = params.get('from_point_key')
        to_model, to_id = params.get('to_point_key')

        from_point = from_model.objects.get(id=from_id)
        to_point = to_model.objects.get(id=to_id)
        from_settlement_or_station = get_settlement_by_point(from_point) or self.station_or_none(from_point)
        to_settlement_or_station = get_settlement_by_point(to_point) or self.station_or_none(to_point)

        has_routes = False

        if (
            from_settlement_or_station and to_settlement_or_station
            and (from_settlement_or_station.id != to_settlement_or_station.id or from_model != to_model)
        ):
            left_date = params.get('left_date')
            right_date = params.get('right_date')

            # Для фильтрации нужно преобразовать в datetime
            # и сделать по часовому поясу города отправления
            until_aware_dt = from_settlement_or_station.localize(
                loc=datetime.combine(right_date, datetime.min.time())
            )

            for s in route_find(from_settlement_or_station, to_settlement_or_station, left_date, 'plane'):
                if s.departure > until_aware_dt:
                    break

                has_routes = True
                break

        return {
            'has_routes': has_routes
        }

    @staticmethod
    def station_or_none(point):
        return point if isinstance(point, Station) else None
