# -*- encoding: utf-8 -*-
from __future__ import absolute_import

from cachetools import cached, TTLCache
import flask
from marshmallow import Schema, fields
from django.conf import settings
from django.db.models import Q
from django.utils.translation import get_language

from travel.avia.library.python.common.models.geo import Country, Region, Settlement, Station, CityMajority
from travel.avia.library.python.common.models.tariffs import Setting
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.utils.caching import cache_until_switch

from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.fields import Related, ModelField, SettlementKey
from travel.avia.backend.main.api.register import get_handler
from travel.avia.backend.main.lib.geo import has_airport, get_capital

from unidecode import unidecode


def _find_settlement(code):
    if unicode(code).isdigit():
        try:
            return Settlement.objects.get(pk=code)
        except Settlement.DoesNotExist:
            return None

    try:
        return Settlement.objects.filter(
            Q(human_url=code) | Q(iata=code) | Q(sirena_id=code)
        ).order_by('majority')[0]
    except IndexError:
        try:
            station = Station.get_by_code('iata', code)
            if station and not station.settlement.hidden:
                return station.settlement
        except Station.DoesNotExist:
            return None


def _fill_settlement(settlement):
    # у нас iata более хитрая вещь
    settlement.iata = get_iata_code(settlement)
    # спорные территории
    settlement.country = get_country(settlement)

    return settlement


def get_settlement(code):
    settlement = _find_settlement(code)

    if settlement:
        _fill_settlement(settlement)

    return settlement


def fill_related_fields(settlements, fill_capital=False):
    country_ids = set([s.country_id for s in settlements])
    countries = {c.id: c for c in list(Country.objects.filter(id__in=country_ids))}

    region_ids = set([s.region_id for s in settlements if s.region_id])
    regions = {r.id: r for r in list(Region.objects.filter(id__in=region_ids))}

    if fill_capital:
        capitals = Settlement.objects.filter(country__in=country_ids,
                                             majority__id=CityMajority.CAPITAL_ID)
        capitals = set(capitals.values_list("id", flat=True))

    for settlement in settlements:
        settlement.region = regions.get(settlement.region_id, None)
        # Устанавливаем страну из кэша, т.к. у get_country fallback на self.country
        settlement.country = countries.get(settlement.country_id, None)
        settlement.country = get_country(settlement)

        if fill_capital:
            settlement.is_capital = settlement.id in capitals


def get_iata_code(settlement):
    code = getattr(settlement, 'iata', None)

    if code:
        return code

    return _get_iata_code_by_stations(settlement)


@cached(cache=TTLCache(maxsize=float('inf'), ttl=30 * 60), key=lambda s: s.id)
def _get_iata_code_by_stations(settlement):
    airports = Station.objects.filter(
        settlement=settlement, hidden=False, t_type__in=[TransportType.PLANE_ID]
    ).distinct()[:1]

    if len(airports):
        code = airports[0].iata
        if code:
            return code

    airports = Station.objects.filter(
        station2settlement__settlement=settlement, hidden=False,
        t_type__in=[TransportType.PLANE_ID]
    ).distinct()[:1]

    if len(airports):
        return airports[0].iata


def get_country(settlement):
    return settlement.translocal_country(flask.g.national_version)


def _safe_convert_to_int(v):
    try:
        return int(v)
    except:
        return None


@cache_until_switch
def has_taxi(city):
    if not city._geo_id:
        return False

    try:
        taxi_geoids = Setting.objects.get(code='TAXI_GEOIDS').value
    except Setting.DoesNotExist:
        taxi_geoids = '213'  # fallback на москву

    taxi_geoids = filter(
        lambda x: x is not None, map(_safe_convert_to_int, taxi_geoids.split(','))
    )

    return city._geo_id in taxi_geoids


def is_capital(city):
    if (hasattr(city, 'is_capital')):
        return city.is_capital

    return get_capital(city.country, flask.g.national_version) == city


def get_url_title(s):
    lang, _ = settings.DOMAIN_LANGUAGE_MAP.get(flask.g.national_version, 'en')
    return unidecode(s.L_title(lang=lang)).replace("'", '')


class SettlementParams(Schema):
    id = fields.Integer()
    key = SettlementKey()
    code = fields.Str()

    settlement = ModelField(model=Settlement, allow_none=True)


class SettlementSchema(TypeSchema):
    id = fields.Integer()
    key = fields.Str(attribute='point_key')
    title = fields.Str(attribute='L_title')
    url_title = fields.Function(get_url_title)
    code = fields.Function(lambda s: get_iata_code(s) or s.sirena_id)
    geo_id = fields.Int(attribute='_geo_id')
    iata_code = fields.Function(lambda s: get_iata_code(s))
    sirena_code = fields.Str(attribute='sirena_id')
    utcoffset = fields.Int()

    country = Related(handler=get_handler('country'), params={'country': 'country'})
    is_capital = fields.Function(is_capital)

    title_genitive = fields.Function(lambda s: s.L_title(case='genitive'))
    title_accusative = fields.Function(lambda s: s.L_title(case='accusative'))
    title_locative = fields.Function(lambda s: s.L_title(case='locative'))
    title_preposition = fields.Function(lambda s: s.L_title(case='preposition_v_vo_na'))
    phrase_from = fields.Method('get_phrase_from')
    phrase_to = fields.Method('get_phrase_to')
    phrase_in = fields.Function(lambda s: s.L_title(case='phrase_in'))

    latitude = fields.Float()
    longitude = fields.Float()

    has_airport = fields.Function(has_airport)
    has_taxi = fields.Function(has_taxi)

    airports = Related(
        handler=get_handler('stations'),
        params={'settlement': 'self'},
        extra_params={'ttype': 'airport'},
    )

    aeroexpress = Related(
        handler=get_handler('stations'),
        params={'settlement': 'self'},
        extra_params={'ttype': 'aeroexpress'},
    )

    trains = Related(
        handler=get_handler('stations'),
        params={'settlement': 'self'},
        extra_params={'ttype': 'train'},
    )

    def get_phrase_to(self, s):
        lang = get_language()

        if lang not in ('ru', 'uk'):
            return s.L_title()

        title = s.L_title(case='accusative', fallback=False)

        if not title:
            return s.L_title()

        return title

    def get_phrase_from(self, s):
        lang = get_language()

        if lang not in ('ru', 'uk'):
            return s.L_title()

        title = s.L_title(case='genitive', fallback=False)

        if not title:
            return s.L_title()

        return title


class SettlementHandler(ApiHandler):
    PARAMS_SCHEMA = SettlementParams
    TYPE_SCHEMA = SettlementSchema

    def preprocess_fields(self, fields):
        if not fields:
            fields = ['id', 'title', 'code']

        return fields

    def process(self, params, fields):
        if params.get('settlement'):
            return _fill_settlement(params.get('settlement'))

        settlement_code = params.get('id') or params.get('code')

        if not settlement_code and params.get('key'):
            key = params.get('key')
            settlement_code = key[1:]

        if not settlement_code:
            return None

        return get_settlement(settlement_code)
